<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Customer;
use App\Models\Service;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ReservationApiController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $query = Reservation::with(['customer', 'service', 'personel', 'room']);

        if ($user->role === 'personel') {
            $query->where('personel_id', $user->id);
        } elseif ($user->role === 'beauty_salon') {
            $query->where('beauty_salon_id', $user->beauty_salon_id);
        } elseif ($user->role === 'randevucu') {
            $query->where('company_id', $user->company_id);
        }

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('date_from')) {
            $query->whereDate('date', '>=', $request->date_from);
        }

        if ($request->has('date_to')) {
            $query->whereDate('date', '<=', $request->date_to);
        }

        if ($request->has('staff_id')) {
            $query->where('personel_id', $request->staff_id);
        }

        $appointments = $query->orderBy('date', 'desc')
            ->orderBy('start_time', 'desc')
            ->paginate($request->get('per_page', 20));

        return response()->json([
            'success' => true,
            'data' => $appointments->items(),
            'meta' => [
                'current_page' => $appointments->currentPage(),
                'last_page' => $appointments->lastPage(),
                'per_page' => $appointments->perPage(),
                'total' => $appointments->total(),
            ],
        ]);
    }

    public function show(Request $request, int $id): JsonResponse
    {
        $user = $request->user();
        
        $appointment = Reservation::with(['customer', 'service', 'personel', 'room'])
            ->find($id);

        if (!$appointment) {
            return response()->json([
                'success' => false,
                'message' => 'Randevu bulunamadı',
            ], 404);
        }

        if ($user->role === 'personel' && $appointment->personel_id !== $user->id) {
            return response()->json([
                'success' => false,
                'message' => 'Bu randevuya erişim yetkiniz yok',
            ], 403);
        }

        return response()->json([
            'success' => true,
            'data' => $this->formatAppointment($appointment),
        ]);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'service_id' => 'required|exists:services,id',
            'date' => 'required|date',
            'start_time' => 'required|date_format:H:i',
            'staff_id' => 'nullable|exists:users,id',
            'room_id' => 'nullable|exists:rooms,id',
            'total_price' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();

        $conflict = $this->checkAvailability(
            $request->date,
            $request->start_time,
            $request->staff_id,
            $request->room_id
        );

        if ($conflict) {
            return response()->json([
                'success' => false,
                'message' => 'Bu saatte başka bir randevu var',
                'code' => 'APPOINTMENT_CONFLICT',
            ], 409);
        }

        try {
            DB::beginTransaction();

            $service = Service::find($request->service_id);
            $duration = $service->duration ?? 60; 

            $endTime = Carbon::createFromFormat('H:i', $request->start_time)
                ->addMinutes($duration)
                ->format('H:i');

            $appointment = Reservation::create([
                'customer_id' => $request->customer_id,
                'beauty_salon_id' => $user->beauty_salon_id ?? $user->id,
                'company_id' => $user->company_id,
                'personel_id' => $request->staff_id,
                'room_id' => $request->room_id,
                'date' => $request->date,
                'start_time' => $request->start_time,
                'end_time' => $endTime,
                'total_price' => $request->total_price ?? $service->price ?? 0,
                'status' => 'onaylandi',
                'notes' => $request->notes,
                'created_by' => $user->id,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Randevu başarıyla oluşturuldu',
                'data' => $this->formatAppointment($appointment->load(['customer', 'service', 'personel', 'room'])),
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'success' => false,
                'message' => 'Randevu oluşturulurken hata oluştu',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function checkAvailableTimes(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'date' => 'required|date',
            'service_id' => 'required|exists:services,id',
            'staff_id' => 'nullable|exists:users,id',
            'room_id' => 'nullable|exists:rooms,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri',
                'errors' => $validator->errors(),
            ], 422);
        }

        $date = Carbon::parse($request->date);
        $service = Service::find($request->service_id);
        $duration = $service->duration ?? 60;

        $workingHours = $this->getWorkingHours($request->user());
        
        $availableSlots = [];
        $currentTime = Carbon::createFromTimeString($workingHours['start']);

        while ($currentTime->format('H:i') < $workingHours['end']) {
            $slotTime = $currentTime->format('H:i');
            $endTime = $currentTime->copy()->addMinutes($duration)->format('H:i');

            $isAvailable = !$this->hasConflict(
                $date->toDateString(),
                $slotTime,
                $endTime,
                $request->staff_id,
                $request->room_id
            );

            $availableSlots[] = [
                'time' => $slotTime,
                'available' => $isAvailable,
                'reason' => $isAvailable ? null : 'Bu saatte başka bir randevu var',
            ];

            $currentTime->addMinutes(30); 
        }

        return response()->json([
            'success' => true,
            'data' => $availableSlots,
        ]);
    }

    public function upcoming(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $query = Reservation::with(['customer', 'service', 'personel', 'room'])
            ->where('status', 'onaylandi')
            ->whereDate('date', '>=', now());

        if ($user->role === 'personel') {
            $query->where('personel_id', $user->id);
        } elseif ($user->role === 'beauty_salon') {
            $query->where('beauty_salon_id', $user->beauty_salon_id);
        }

        $appointments = $query->orderBy('date', 'asc')
            ->orderBy('start_time', 'asc')
            ->limit(10)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $appointments->map(fn($apt) => $this->formatAppointment($apt)),
        ]);
    }

    private function checkAvailability(string $date, string $startTime, ?int $staffId, ?int $roomId): bool
    {
        return $this->hasConflict($date, $startTime, null, $staffId, $roomId);
    }

    private function hasConflict(string $date, string $startTime, ?string $endTime, ?int $staffId, ?int $roomId): bool
    {
        $query = Reservation::where('status', '!=', 'iptal')
            ->whereDate('date', $date)
            ->where(function ($q) use ($startTime, $endTime) {
                $q->whereBetween('start_time', [$startTime, $endTime ?? $startTime])
                  ->orWhereBetween('end_time', [$startTime, $endTime ?? $startTime])
                  ->orWhere(function ($q2) use ($startTime, $endTime) {
                      $q2->where('start_time', '<=', $startTime)
                         ->where('end_time', '>=', $endTime ?? $startTime);
                  });
            });

        if ($staffId) {
            $query->where('personel_id', $staffId);
        }

        if ($roomId) {
            $query->where('room_id', $roomId);
        }

        return $query->exists();
    }

    private function getWorkingHours($user): array
    {
        
        $settings = \App\Models\Setting::where('user_id', $user->id)
            ->where('group', 'calendar')
            ->pluck('value', 'key')
            ->toArray();

        return [
            'start' => $settings['working_hours_start'] ?? '09:00',
            'end' => $settings['working_hours_end'] ?? '18:00',
        ];
    }

    private function formatAppointment(Reservation $appointment): array
    {
        return [
            'id' => $appointment->id,
            'customer_id' => $appointment->customer_id,
            'customer_name' => $appointment->customer->name ?? null,
            'customer_phone' => $appointment->customer->phone ?? null,
            'service_id' => $appointment->service->id ?? null,
            'service_name' => $appointment->service->name ?? null,
            'staff_id' => $appointment->personel_id,
            'staff_name' => $appointment->personel->name ?? null,
            'room_id' => $appointment->room_id,
            'room_name' => $appointment->room->name ?? null,
            'date' => $appointment->date->format('Y-m-d'),
            'start_time' => $appointment->start_time,
            'end_time' => $appointment->end_time,
            'status' => $appointment->status,
            'total_price' => (float) $appointment->total_price,
            'deposit' => (float) $appointment->deposit_paid,
            'notes' => $appointment->notes,
            'created_at' => $appointment->created_at?->toISOString(),
            'updated_at' => $appointment->updated_at?->toISOString(),
        ];
    }
}

