<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Http\Resources\Api\AppointmentResource;
use Illuminate\Http\Request;
use App\Models\Reservation;
use App\Models\Customer;
use Carbon\Carbon;

class AppointmentController extends Controller
{
    use ApiResponse;

    /**
     * Display a listing of appointments
     */
    public function index(Request $request)
    {
        $user = $request->user();
        
        $query = Reservation::where('beauty_salon_id', $user->id)
            ->with(['customer', 'personel', 'room', 'services']);
        
        // Filters
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }
        
        if ($request->has('date')) {
            $query->whereDate('date', $request->date);
        }
        
        if ($request->has('date_from')) {
            $query->where('date', '>=', $request->date_from);
        }
        
        if ($request->has('date_to')) {
            $query->where('date', '<=', $request->date_to);
        }
        
        $appointments = $query->orderBy('date', 'desc')
            ->orderBy('start_time', 'desc')
            ->paginate($request->get('per_page', 15));
        
        return $this->paginatedResponse($appointments, 'Appointments retrieved successfully');
    }

    /**
     * Get upcoming appointments
     */
    public function upcoming(Request $request)
    {
        $user = $request->user();
        
        $appointments = Reservation::where('beauty_salon_id', $user->id)
            ->where('date', '>=', Carbon::today())
            ->whereIn('status', ['confirmed', 'pending'])
            ->with(['customer', 'personel', 'room', 'services'])
            ->orderBy('date')
            ->orderBy('start_time')
            ->limit(20)
            ->get();
        
        return $this->successResponse(
            AppointmentResource::collection($appointments),
            'Upcoming appointments retrieved successfully'
        );
    }

    /**
     * Store a newly created appointment
     */
    public function store(Request $request)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'date' => 'required|date',
            'start_time' => 'required|date_format:H:i',
            'personel_id' => 'nullable|exists:users,id',
            'room_id' => 'nullable|exists:rooms,id',
            'service_ids' => 'required|array',
            'service_ids.*' => 'exists:services,id',
            'total_amount' => 'required|numeric|min:0',
            'deposit_amount' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        $user = $request->user();
        
        $appointment = Reservation::create([
            'beauty_salon_id' => $user->id,
            'customer_id' => $request->customer_id,
            'date' => $request->date,
            'start_time' => $request->start_time,
            'end_time' => Carbon::parse($request->start_time)->addMinutes(30)->format('H:i'),
            'personel_id' => $request->personel_id,
            'room_id' => $request->room_id,
            'total_amount' => $request->total_amount,
            'deposit_amount' => $request->deposit_amount ?? 0,
            'notes' => $request->notes,
            'status' => 'confirmed',
        ]);

        $appointment->services()->attach($request->service_ids);

        return $this->successResponse(
            new AppointmentResource($appointment->load(['customer', 'personel', 'room', 'services'])),
            'Appointment created successfully',
            201
        );
    }

    /**
     * Display the specified appointment
     */
    public function show($id)
    {
        $appointment = Reservation::with(['customer', 'personel', 'room', 'services'])
            ->findOrFail($id);
        
        return $this->successResponse(
            new AppointmentResource($appointment),
            'Appointment retrieved successfully'
        );
    }

    /**
     * Update the specified appointment
     */
    public function update(Request $request, $id)
    {
        $appointment = Reservation::findOrFail($id);
        
        $request->validate([
            'customer_id' => 'sometimes|exists:customers,id',
            'date' => 'sometimes|date',
            'start_time' => 'sometimes|date_format:H:i',
            'personel_id' => 'nullable|exists:users,id',
            'room_id' => 'nullable|exists:rooms,id',
            'service_ids' => 'sometimes|array',
            'service_ids.*' => 'exists:services,id',
            'total_amount' => 'sometimes|numeric|min:0',
            'deposit_amount' => 'nullable|numeric|min:0',
            'notes' => 'nullable|string',
            'status' => 'sometimes|in:pending,confirmed,cancelled,completed',
        ]);

        $appointment->update($request->only([
            'customer_id', 'date', 'start_time', 'personel_id', 'room_id',
            'total_amount', 'deposit_amount', 'notes', 'status'
        ]));

        if ($request->has('start_time')) {
            $appointment->end_time = Carbon::parse($request->start_time)->addMinutes(30)->format('H:i');
            $appointment->save();
        }

        if ($request->has('service_ids')) {
            $appointment->services()->sync($request->service_ids);
        }

        return $this->successResponse(
            new AppointmentResource($appointment->load(['customer', 'personel', 'room', 'services'])),
            'Appointment updated successfully'
        );
    }

    /**
     * Remove the specified appointment
     */
    public function destroy($id)
    {
        $appointment = Reservation::findOrFail($id);
        $appointment->delete();

        return $this->successResponse(null, 'Appointment deleted successfully');
    }

    /**
     * Approve appointment
     */
    public function approve($id)
    {
        $appointment = Reservation::findOrFail($id);
        $appointment->update(['status' => 'confirmed']);

        return $this->successResponse(
            new AppointmentResource($appointment->load(['customer', 'personel', 'room', 'services'])),
            'Appointment approved successfully'
        );
    }

    /**
     * Cancel appointment
     */
    public function cancel($id)
    {
        $appointment = Reservation::findOrFail($id);
        $appointment->update(['status' => 'cancelled']);

        return $this->successResponse(
            new AppointmentResource($appointment->load(['customer', 'personel', 'room', 'services'])),
            'Appointment cancelled successfully'
        );
    }

    /**
     * Complete appointment
     */
    public function complete($id)
    {
        $appointment = Reservation::findOrFail($id);
        $appointment->update(['status' => 'completed']);

        return $this->successResponse(
            new AppointmentResource($appointment->load(['customer', 'personel', 'room', 'services'])),
            'Appointment completed successfully'
        );
    }

    /**
     * Check availability
     */
    public function checkAvailability(Request $request)
    {
        $request->validate([
            'date' => 'required|date',
            'personel_id' => 'nullable|exists:users,id',
            'duration' => 'required|integer|min:1',
        ]);

        // Implementation for checking available time slots
        // This is a simplified version
        
        return $this->successResponse([
            'available_slots' => [
                '09:00', '09:30', '10:00', '10:30', '11:00', '11:30',
                '12:00', '12:30', '13:00', '13:30', '14:00', '14:30',
                '15:00', '15:30', '16:00', '16:30', '17:00', '17:30',
            ],
        ], 'Availability checked successfully');
    }
}

