<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Models\AppointmentRequest;
use Illuminate\Http\Request;

class AppointmentRequestController extends Controller
{
    use ApiResponse;

    public function index(Request $request)
    {
        $user = $request->user();
        
        $query = AppointmentRequest::where('beauty_salon_id', $user->id)
            ->with(['service', 'customer', 'personel', 'room']);
        
        if ($request->has('status')) {
            $query->where('status', $request->status);
        } else {
            // Default to pending
            $query->where('status', 'pending');
        }
        
        $requests = $query->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 15));
        
        return $this->paginatedResponse($requests, 'Appointment requests retrieved successfully');
    }

    public function show(Request $request, $id)
    {
        $user = $request->user();
        $appointmentRequest = AppointmentRequest::with(['service', 'customer', 'personel', 'room'])
            ->find($id);

        if (!$appointmentRequest) {
            return $this->notFoundResponse('Randevu talebi bulunamadı.');
        }

        if ($appointmentRequest->beauty_salon_id !== $user->id) {
            return $this->forbiddenResponse('Bu randevu talebine erişim yetkiniz yok.');
        }

        return $this->successResponse([
            'id' => $appointmentRequest->id,
            'name' => $appointmentRequest->name,
            'surname' => $appointmentRequest->surname,
            'phone' => $appointmentRequest->phone,
            'service' => $appointmentRequest->service ? [
                'id' => $appointmentRequest->service->id,
                'name' => $appointmentRequest->service->name,
                'price' => (float) $appointmentRequest->service->price,
            ] : null,
            'status' => $appointmentRequest->status,
            'appointment_date' => $appointmentRequest->appointment_date?->format('Y-m-d'),
            'appointment_time' => $appointmentRequest->appointment_time,
            'notes' => $appointmentRequest->notes,
            'created_at' => $appointmentRequest->created_at?->toISOString(),
        ], 'Randevu talebi detayları.');
    }

    public function approve(Request $request, $id)
    {
        $user = $request->user();
        $appointmentRequest = AppointmentRequest::where('id', $id)
            ->where('beauty_salon_id', $user->id)
            ->where('status', 'pending')
            ->firstOrFail();

        $validated = $request->validate([
            'appointment_date' => 'required|date',
            'appointment_time' => 'required|date_format:H:i',
            'personel_id' => 'nullable|exists:users,id',
            'room_id' => 'nullable|exists:rooms,id',
            'total_price' => 'required|numeric|min:0',
        ]);

        // This would typically call the existing approval logic
        // For now, just return success
        return $this->successResponse(null, 'Randevu talebi onaylandı.');
    }
}

