<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Models\Account;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Carbon\Carbon;

class CashboxController extends Controller
{
    use ApiResponse;

    public function index(Request $request)
    {
        $user = $request->user();
        
        $cashAccount = Account::getCashAccount($user->id);
        
        $baseQuery = Transaction::where('account_id', $cashAccount->id);
        
        // Calculate totals
        $totalIncome = (clone $baseQuery)->where('type', 'income')->sum('amount') ?? 0;
        $totalExpense = (clone $baseQuery)->where('type', 'expense')->sum('amount') ?? 0;
        
        $today = Carbon::today();
        $todayIncome = (clone $baseQuery)->where('type', 'income')->whereDate('date', $today)->sum('amount') ?? 0;
        $todayExpense = (clone $baseQuery)->where('type', 'expense')->whereDate('date', $today)->sum('amount') ?? 0;
        
        $monthStart = Carbon::now()->startOfMonth();
        $monthEnd = Carbon::now()->endOfMonth();
        $monthlyIncome = (clone $baseQuery)->where('type', 'income')->whereBetween('date', [$monthStart, $monthEnd])->sum('amount') ?? 0;
        $monthlyExpense = (clone $baseQuery)->where('type', 'expense')->whereBetween('date', [$monthStart, $monthEnd])->sum('amount') ?? 0;
        
        $totalBalance = $totalIncome - $totalExpense;
        
        // Get transactions
        $query = Transaction::where('account_id', $cashAccount->id)
            ->whereIn('type', ['income', 'expense'])
            ->with(['account', 'reservation', 'customer']);
        
        if ($request->has('date_from')) {
            $query->whereDate('date', '>=', $request->date_from);
        }
        
        if ($request->has('date_to')) {
            $query->whereDate('date', '<=', $request->date_to);
        }
        
        if ($request->has('type') && in_array($request->type, ['income', 'expense'])) {
            $query->where('type', $request->type);
        }
        
        $transactions = $query->orderBy('date', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 20));
        
        $formattedTransactions = $transactions->map(function ($transaction) {
            return [
                'id' => $transaction->id,
                'type' => $transaction->type,
                'amount' => (float) $transaction->amount,
                'description' => $transaction->description,
                'date' => $transaction->date?->format('Y-m-d') ?? $transaction->created_at?->format('Y-m-d'),
            ];
        });
        
        return $this->successResponse([
            'stats' => [
                'total_income' => (float) $totalIncome,
                'total_expense' => (float) $totalExpense,
                'total_balance' => (float) $totalBalance,
                'today_income' => (float) $todayIncome,
                'today_expense' => (float) $todayExpense,
                'monthly_income' => (float) $monthlyIncome,
                'monthly_expense' => (float) $monthlyExpense,
            ],
            'transactions' => $formattedTransactions,
            'meta' => [
                'pagination' => [
                    'current_page' => $transactions->currentPage(),
                    'last_page' => $transactions->lastPage(),
                    'per_page' => $transactions->perPage(),
                    'total' => $transactions->total(),
                ],
            ],
        ], 'Cashbox data retrieved successfully');
    }
    
    public function storeIncome(Request $request)
    {
        $user = $request->user();
        
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'date' => 'required|date',
            'description' => 'nullable|string',
        ]);
        
        $cashAccount = \App\Models\Account::getCashAccount($user->id);
        
        $transaction = \App\Models\Transaction::create([
            'account_id' => $cashAccount->id,
            'type' => 'income',
            'amount' => $validated['amount'],
            'date' => $validated['date'],
            'description' => $validated['description'] ?? 'Gelir',
            'created_by' => $user->id,
        ]);
        
        return $this->successResponse([
            'id' => $transaction->id,
            'type' => $transaction->type,
            'amount' => (float) $transaction->amount,
            'description' => $transaction->description,
            'date' => $transaction->date?->format('Y-m-d'),
        ], 'Gelir eklendi', 201);
    }
    
    public function storeExpense(Request $request)
    {
        $user = $request->user();
        
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'date' => 'required|date',
            'description' => 'nullable|string',
        ]);
        
        $cashAccount = \App\Models\Account::getCashAccount($user->id);
        
        $transaction = \App\Models\Transaction::create([
            'account_id' => $cashAccount->id,
            'type' => 'expense',
            'amount' => $validated['amount'],
            'date' => $validated['date'],
            'description' => $validated['description'] ?? 'Gider',
            'created_by' => $user->id,
        ]);
        
        return $this->successResponse([
            'id' => $transaction->id,
            'type' => $transaction->type,
            'amount' => (float) $transaction->amount,
            'description' => $transaction->description,
            'date' => $transaction->date?->format('Y-m-d'),
        ], 'Gider eklendi', 201);
    }
}

