<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Http\Resources\Api\CustomerResource;
use Illuminate\Http\Request;
use App\Models\Customer;

class CustomerController extends Controller
{
    use ApiResponse;

    /**
     * Display a listing of customers
     */
    public function index(Request $request)
    {
        $user = $request->user();
        
        $query = Customer::where('beauty_salon_id', $user->id);
        
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }
        
        $customers = $query->orderBy('name')
            ->paginate($request->get('per_page', 15));
        
        return $this->paginatedResponse($customers, 'Customers retrieved successfully');
    }

    /**
     * Search customers
     */
    public function search(Request $request)
    {
        $request->validate([
            'query' => 'required|string|min:2',
        ]);

        $user = $request->user();
        
        $customers = Customer::where('beauty_salon_id', $user->id)
            ->where(function ($q) use ($request) {
                $q->where('name', 'like', "%{$request->query}%")
                  ->orWhere('phone', 'like', "%{$request->query}%")
                  ->orWhere('email', 'like', "%{$request->query}%");
            })
            ->limit(20)
            ->get();
        
        return $this->successResponse(
            CustomerResource::collection($customers),
            'Search results retrieved successfully'
        );
    }

    /**
     * Store a newly created customer
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string|max:500',
        ]);

        $user = $request->user();
        
        $customer = Customer::create([
            'beauty_salon_id' => $user->id,
            'name' => $request->name,
            'phone' => $request->phone,
            'email' => $request->email,
            'address' => $request->address,
        ]);

        return $this->successResponse(
            new CustomerResource($customer),
            'Customer created successfully',
            201
        );
    }

    /**
     * Display the specified customer
     */
    public function show($id)
    {
        $customer = Customer::findOrFail($id);
        
        return $this->successResponse(
            new CustomerResource($customer),
            'Customer retrieved successfully'
        );
    }

    /**
     * Update the specified customer
     */
    public function update(Request $request, $id)
    {
        $customer = Customer::findOrFail($id);
        
        $request->validate([
            'name' => 'sometimes|string|max:255',
            'phone' => 'sometimes|string|max:20',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string|max:500',
        ]);

        $customer->update($request->only(['name', 'phone', 'email', 'address']));

        return $this->successResponse(
            new CustomerResource($customer),
            'Customer updated successfully'
        );
    }

    /**
     * Remove the specified customer
     */
    public function destroy($id)
    {
        $customer = Customer::findOrFail($id);
        $customer->delete();

        return $this->successResponse(null, 'Customer deleted successfully');
    }
}

