<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use Illuminate\Http\Request;
use App\Models\Reservation;
use App\Models\Customer;
use Carbon\Carbon;

class DashboardController extends Controller
{
    use ApiResponse;

    /**
     * Get dashboard data
     */
    public function index(Request $request)
    {
        $user = $request->user();
        
        $today = Carbon::today();
        $thisMonth = Carbon::now()->startOfMonth();
        
        // Today's appointments
        $todayAppointments = Reservation::where('beauty_salon_id', $user->id)
            ->whereDate('date', $today)
            ->count();
        
        // This month's appointments
        $monthAppointments = Reservation::where('beauty_salon_id', $user->id)
            ->where('date', '>=', $thisMonth)
            ->count();
        
        // Total customers
        $totalCustomers = Customer::where('beauty_salon_id', $user->id)->count();
        
        // Upcoming appointments (next 7 days)
        $upcomingAppointments = Reservation::where('beauty_salon_id', $user->id)
            ->where('date', '>=', $today)
            ->where('date', '<=', $today->copy()->addDays(7))
            ->whereIn('status', ['confirmed', 'pending', 'open'])
            ->with(['customer', 'personel', 'room', 'services'])
            ->orderBy('date')
            ->orderBy('start_time')
            ->limit(10)
            ->get();
        
        // Recent appointments (last 10)
        $recentAppointments = Reservation::where('beauty_salon_id', $user->id)
            ->with(['customer', 'personel', 'room', 'services'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
        
        // Recent customers (last 10)
        $recentCustomers = Customer::where('beauty_salon_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
        
        // Today's revenue
        $todayRevenue = Reservation::where('beauty_salon_id', $user->id)
            ->whereDate('date', $today)
            ->where('status', 'completed')
            ->sum('total_amount') ?? 0;
        
        // Monthly revenue
        $monthlyRevenue = Reservation::where('beauty_salon_id', $user->id)
            ->where('date', '>=', $thisMonth)
            ->where('status', 'completed')
            ->sum('total_amount') ?? 0;
        
        return $this->successResponse([
            'stats' => [
                'today_appointments' => $todayAppointments,
                'month_appointments' => $monthAppointments,
                'total_customers' => $totalCustomers,
                'today_revenue' => (float) $todayRevenue,
                'monthly_revenue' => (float) $monthlyRevenue,
            ],
            'upcoming_appointments' => \App\Http\Resources\Api\AppointmentResource::collection($upcomingAppointments),
            'recent_appointments' => \App\Http\Resources\Api\AppointmentResource::collection($recentAppointments),
            'recent_customers' => \App\Http\Resources\Api\CustomerResource::collection($recentCustomers),
        ]);
    }
}

