<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Models\Invoice;
use Illuminate\Http\Request;

class InvoiceController extends Controller
{
    use ApiResponse;

    public function index(Request $request)
    {
        $user = $request->user();
        
        $query = Invoice::with(['customer', 'reservation'])
            ->where('beauty_salon_id', $user->id);
        
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }
        
        if ($request->has('date_from')) {
            $query->whereDate('invoice_date', '>=', $request->date_from);
        }
        
        if ($request->has('date_to')) {
            $query->whereDate('invoice_date', '<=', $request->date_to);
        }
        
        $invoices = $query->orderBy('invoice_date', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 15));
        
        $formattedInvoices = $invoices->map(function ($invoice) {
            return [
                'id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number,
                'customer_name' => $invoice->customer ? $invoice->customer->name : null,
                'total_amount' => (float) $invoice->total_amount,
                'paid_amount' => (float) ($invoice->paid_amount ?? 0),
                'remaining_amount' => (float) ($invoice->remaining_amount ?? $invoice->total_amount),
                'status' => $invoice->status,
                'invoice_date' => $invoice->invoice_date?->format('Y-m-d'),
            ];
        });
        
        return $this->successResponse([
            'data' => $formattedInvoices,
            'meta' => [
                'pagination' => [
                    'current_page' => $invoices->currentPage(),
                    'last_page' => $invoices->lastPage(),
                    'per_page' => $invoices->perPage(),
                    'total' => $invoices->total(),
                ],
            ],
        ], 'Invoices retrieved successfully');
    }

    public function show(Request $request, $id)
    {
        $user = $request->user();
        $invoice = Invoice::with(['customer', 'reservation', 'items'])->find($id);

        if (!$invoice) {
            return $this->notFoundResponse('Fatura bulunamadı.');
        }

        if ($invoice->beauty_salon_id !== $user->id) {
            return $this->forbiddenResponse('Bu faturaya erişim yetkiniz yok.');
        }

        return $this->successResponse([
            'id' => $invoice->id,
            'invoice_number' => $invoice->invoice_number,
            'customer' => $invoice->customer ? [
                'id' => $invoice->customer->id,
                'name' => $invoice->customer->name,
                'phone' => $invoice->customer->phone,
            ] : null,
            'total_amount' => (float) $invoice->total_amount,
            'paid_amount' => (float) $invoice->paid_amount,
            'remaining_amount' => (float) $invoice->remaining_amount,
            'status' => $invoice->status,
            'invoice_date' => $invoice->invoice_date?->format('Y-m-d'),
            'items' => $invoice->items ? json_decode($invoice->items, true) : [],
            'created_at' => $invoice->created_at?->toISOString(),
        ], 'Fatura detayları.');
    }
}

