<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Models\Reservation;
use Carbon\Carbon;
use Illuminate\Http\Request;

class NotificationController extends Controller
{
    use ApiResponse;

    public function index(Request $request)
    {
        $user = $request->user();
        $notifications = [];

        // Yeni randevular (son 7 gün)
        $newAppointments = Reservation::where('beauty_salon_id', $user->id)
            ->where('created_at', '>=', Carbon::now()->subDays(7))
            ->with(['customer', 'personel', 'room'])
            ->orderBy('created_at', 'desc')
            ->limit(20)
            ->get();
            
        foreach ($newAppointments as $reservation) {
            $notifications[] = [
                'id' => $reservation->id,
                'type' => 'new_reservation',
                'icon' => 'calendar-check',
                'color' => 'primary',
                'title' => 'Yeni randevu oluşturuldu',
                'message' => ($reservation->customer ? $reservation->customer->name : 'Müşteri') . ' için randevu oluşturuldu',
                'date' => $reservation->created_at->toISOString(),
                'appointment_id' => $reservation->id,
            ];
        }

        // Yaklaşan randevular (3 gün içinde)
        $upcomingAppointments = Reservation::where('beauty_salon_id', $user->id)
            ->whereBetween('date', [
                Carbon::now()->toDateString(),
                Carbon::now()->addDays(3)->toDateString()
            ])
            ->where('status', '!=', 'cancelled')
            ->with(['customer', 'personel', 'room'])
            ->orderBy('date', 'asc')
            ->orderBy('start_time', 'asc')
            ->limit(20)
            ->get();
            
        foreach ($upcomingAppointments as $reservation) {
            $notifications[] = [
                'id' => 'upcoming_' . $reservation->id,
                'type' => 'upcoming',
                'icon' => 'clock',
                'color' => 'warning',
                'title' => 'Yaklaşan randevu',
                'message' => ($reservation->customer ? $reservation->customer->name : 'Müşteri') . ' - ' . Carbon::parse($reservation->date)->format('d.m.Y') . ' ' . $reservation->start_time,
                'date' => Carbon::parse($reservation->date . ' ' . $reservation->start_time)->toISOString(),
                'appointment_id' => $reservation->id,
            ];
        }

        // Tarihe göre sırala
        usort($notifications, function($a, $b) {
            return strtotime($b['date']) <=> strtotime($a['date']);
        });

        // Limit to 30 notifications
        $notifications = array_slice($notifications, 0, 30);
        
        return $this->successResponse($notifications, 'Notifications retrieved successfully');
    }
}


