<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use Illuminate\Http\Request;
use App\Models\Room;

class RoomController extends Controller
{
    use ApiResponse;

    /**
     * Display a listing of rooms
     */
    public function index(Request $request)
    {
        $user = $request->user();
        
        $rooms = Room::where('beauty_salon_id', $user->id)
            ->orderBy('name')
            ->get();
        
        return $this->successResponse(
            $rooms->map(function ($room) {
                return [
                    'id' => $room->id,
                    'name' => $room->name,
                    'description' => $room->description,
                    'tags' => $room->tags,
                    'created_at' => $room->created_at?->toISOString(),
                    'updated_at' => $room->updated_at?->toISOString(),
                ];
            }),
            'Rooms retrieved successfully'
        );
    }

    /**
     * Store a newly created room
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'tags' => 'nullable|string',
        ]);

        $user = $request->user();
        
        $room = Room::create([
            'beauty_salon_id' => $user->id,
            'name' => $request->name,
            'description' => $request->description,
            'tags' => $request->tags,
        ]);

        return $this->successResponse([
            'id' => $room->id,
            'name' => $room->name,
            'description' => $room->description,
            'tags' => $room->tags,
        ], 'Room created successfully', 201);
    }

    /**
     * Display the specified room
     */
    public function show($id)
    {
        $room = Room::findOrFail($id);
        
        return $this->successResponse([
            'id' => $room->id,
            'name' => $room->name,
            'description' => $room->description,
            'tags' => $room->tags,
        ], 'Room retrieved successfully');
    }

    /**
     * Update the specified room
     */
    public function update(Request $request, $id)
    {
        $room = Room::findOrFail($id);
        
        $request->validate([
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'tags' => 'nullable|string',
        ]);

        $room->update($request->only(['name', 'description', 'tags']));

        return $this->successResponse([
            'id' => $room->id,
            'name' => $room->name,
            'description' => $room->description,
            'tags' => $room->tags,
        ], 'Room updated successfully');
    }

    /**
     * Remove the specified room
     */
    public function destroy($id)
    {
        $room = Room::findOrFail($id);
        $room->delete();

        return $this->successResponse(null, 'Room deleted successfully');
    }
}

