<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Http\Resources\Api\ServiceResource;
use Illuminate\Http\Request;
use App\Models\Service;

class ServiceController extends Controller
{
    use ApiResponse;

    /**
     * Display a listing of services
     */
    public function index(Request $request)
    {
        $user = $request->user();
        
        $query = Service::where('beauty_salon_id', $user->id)
            ->with('category');
        
        if ($request->has('category_id')) {
            $query->where('service_category_id', $request->category_id);
        }
        
        $services = $query->orderBy('name')
            ->paginate($request->get('per_page', 15));
        
        return $this->paginatedResponse($services, 'Services retrieved successfully');
    }

    /**
     * Store a newly created service
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'duration' => 'required|integer|min:1',
            'category_id' => 'nullable|exists:service_categories,id',
        ]);

        $user = $request->user();
        
        $service = Service::create([
            'beauty_salon_id' => $user->id,
            'name' => $request->name,
            'description' => $request->description,
            'price' => $request->price,
            'duration' => $request->duration,
            'service_category_id' => $request->category_id,
        ]);

        return $this->successResponse(
            new ServiceResource($service->load('category')),
            'Service created successfully',
            201
        );
    }

    /**
     * Display the specified service
     */
    public function show($id)
    {
        $service = Service::with('category')->findOrFail($id);
        
        return $this->successResponse(
            new ServiceResource($service),
            'Service retrieved successfully'
        );
    }

    /**
     * Update the specified service
     */
    public function update(Request $request, $id)
    {
        $service = Service::findOrFail($id);
        
        $request->validate([
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'price' => 'sometimes|numeric|min:0',
            'duration' => 'sometimes|integer|min:1',
            'category_id' => 'nullable|exists:service_categories,id',
        ]);

        $updateData = $request->only(['name', 'description', 'price', 'duration']);
        if ($request->has('category_id')) {
            $updateData['service_category_id'] = $request->category_id;
        }
        $service->update($updateData);

        return $this->successResponse(
            new ServiceResource($service->load('category')),
            'Service updated successfully'
        );
    }

    /**
     * Remove the specified service
     */
    public function destroy($id)
    {
        $service = Service::findOrFail($id);
        $service->delete();

        return $this->successResponse(null, 'Service deleted successfully');
    }
}

