<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Http\Resources\Api\UserResource;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Reservation;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;

class StaffController extends Controller
{
    use ApiResponse;

    /**
     * Display a listing of staff
     */
    public function index(Request $request)
    {
        $user = $request->user();
        
        $staff = User::where('beauty_salon_id', $user->id)
            ->where('role', 'personel')
            ->orderBy('name')
            ->get();
        
        return $this->successResponse(
            UserResource::collection($staff),
            'Staff retrieved successfully'
        );
    }

    /**
     * Display the specified staff
     */
    public function show($id)
    {
        $staff = User::where('role', 'personel')->findOrFail($id);
        
        return $this->successResponse(
            new UserResource($staff),
            'Staff retrieved successfully'
        );
    }

    /**
     * Get staff schedule
     */
    public function schedule(Request $request, $id)
    {
        $staff = User::where('role', 'personel')->findOrFail($id);
        
        $date = $request->get('date', Carbon::today()->format('Y-m-d'));
        
        $appointments = Reservation::where('personel_id', $id)
            ->whereDate('date', $date)
            ->with(['customer', 'services'])
            ->orderBy('start_time')
            ->get();
        
        return $this->successResponse([
            'staff' => new UserResource($staff),
            'date' => $date,
            'appointments' => $appointments->map(function ($appointment) {
                return [
                    'id' => $appointment->id,
                    'customer_name' => $appointment->customer->name ?? 'N/A',
                    'start_time' => $appointment->start_time?->format('H:i'),
                    'end_time' => $appointment->end_time?->format('H:i'),
                    'status' => $appointment->status,
                    'services' => $appointment->services->pluck('name')->toArray(),
                ];
            }),
        ], 'Schedule retrieved successfully');
    }

    /**
     * Store a newly created staff member
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'nullable|string|max:20',
            'password' => 'nullable|string|min:6',
            'service_ids' => 'nullable|array',
            'service_ids.*' => 'exists:services,id',
            'is_personel_admin' => 'nullable|boolean',
        ]);

        $user = $request->user();
        
        $password = $request->password ?? 'personel123';
        
        $staff = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => Hash::make($password),
            'role' => 'personel',
            'beauty_salon_id' => $user->id,
            'is_personel_admin' => $request->boolean('is_personel_admin', false),
        ]);

        if ($request->has('service_ids') && is_array($request->service_ids)) {
            $staff->services()->sync($request->service_ids);
        }

        return $this->successResponse(
            new UserResource($staff->load('services')),
            'Staff created successfully',
            201
        );
    }

    /**
     * Update the specified staff member
     */
    public function update(Request $request, $id)
    {
        $staff = User::where('role', 'personel')
            ->where('beauty_salon_id', $request->user()->id)
            ->findOrFail($id);
        
        $request->validate([
            'name' => 'sometimes|string|max:255',
            'email' => 'sometimes|email|unique:users,email,' . $id,
            'phone' => 'nullable|string|max:20',
            'password' => 'nullable|string|min:6',
            'service_ids' => 'nullable|array',
            'service_ids.*' => 'exists:services,id',
            'is_personel_admin' => 'nullable|boolean',
        ]);

        $updateData = $request->only(['name', 'phone']);
        
        if ($request->has('email')) {
            $updateData['email'] = $request->email;
        }
        
        if ($request->has('password')) {
            $updateData['password'] = Hash::make($request->password);
        }
        
        if ($request->has('is_personel_admin')) {
            $updateData['is_personel_admin'] = $request->boolean('is_personel_admin');
        }

        $staff->update($updateData);

        if ($request->has('service_ids')) {
            $staff->services()->sync($request->service_ids ?? []);
        }

        return $this->successResponse(
            new UserResource($staff->load('services')),
            'Staff updated successfully'
        );
    }

    /**
     * Remove the specified staff member
     */
    public function destroy($id)
    {
        $staff = User::where('role', 'personel')->findOrFail($id);
        $staff->delete();

        return $this->successResponse(null, 'Staff deleted successfully');
    }
}

