<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class LoginController extends Controller
{
    public function showLoginForm()
    {
        if (Auth::check()) {
            return redirect()->route('admin.dashboard');
        }
        
        return view('auth.login');
    }

    public function login(Request $request)
    {
        $request->validate([
            'login' => 'required|string',
            'password' => 'required|string',
        ]);

        $remember = $request->filled('remember');
        $login = $request->input('login');
        $password = $request->input('password');

        $field = filter_var($login, FILTER_VALIDATE_EMAIL) ? 'email' : 'phone';

        if ($field === 'phone') {
            $login = preg_replace('/[^0-9]/', '', $login);
            if (!empty($login) && !str_starts_with($login, '0')) {
                $login = '0' . $login;
            }
        }

        $user = User::where($field, $login)->first();

        if ($user && Hash::check($password, $user->password)) {
            
            Auth::login($user, $remember);
            $request->session()->regenerate();

            if (!$user->is_active) {
                Auth::logout();
                return back()->withErrors([
                    'login' => 'Hesabınız aktif değil. Lütfen yönetici ile iletişime geçin.',
                ])->onlyInput('login');
            }

            return redirect()->intended(route('admin.dashboard'));
        }

        return back()->withErrors([
            'login' => 'Giriş bilgileri hatalı. E-posta/telefon veya şifreniz yanlış olabilir. Lütfen kontrol edip tekrar deneyin.',
        ])->onlyInput('login');
    }

    public function logout(Request $request)
    {
        Auth::logout();

        try {
            if ($request->hasSession()) {
                $request->session()->invalidate();
                $request->session()->regenerateToken();
            }
        } catch (\Throwable $e) {
            // Sessiyon temizlenemezse bile devam et
        }

        // Ham yönlendirme - view/session bağımlılığı yok, direkt ana sayfaya git
        $homeUrl = '/';
        try {
            $homeUrl = rtrim(config('app.url', ''), '/') ?: '/';
        } catch (\Throwable $e) {
            // config yüklenemezse / kullan
        }

        return response('', 302)->header('Location', $homeUrl);
    }
}
