<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class MobileAuthController extends Controller
{
    public function showLoginForm()
    {
        if (Auth::check()) {
            return redirect()->route('mobile.dashboard');
        }
        
        return view('admin.mobile.auth.login');
    }

    public function login(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required|string',
        ]);

        if (Auth::attempt($credentials, $request->filled('remember'))) {
            $request->session()->regenerate();

            $user = Auth::user();
            
            if (!$user->is_active) {
                Auth::logout();
                return back()->withErrors([
                    'email' => 'Hesabınız aktif değil.',
                ])->onlyInput('email');
            }

            return redirect()->intended(route('mobile.dashboard'));
        }

        throw ValidationException::withMessages([
            'email' => 'E-posta veya şifre hatalı.',
        ]);
    }

    public function showRegisterForm()
    {
        if (Auth::check()) {
            return redirect()->route('mobile.dashboard');
        }
        
        return view('admin.mobile.auth.register');
    }

    public function register(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'required|string|unique:users,phone',
            'password' => 'required|string|min:8|confirmed',
            'pricing_notice_approval' => 'accepted',
        ]);

        $trialEndsAt = now()->addDays(14);

        $phone = preg_replace('/[^0-9]/', '', $validated['phone']);
        if (!empty($phone) && !str_starts_with($phone, '0')) {
            $phone = '0' . $phone;
        }

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $phone,
            'password' => Hash::make($validated['password']),
            'role' => 'güzellik_salonu',
            'is_active' => true,
            'trial_ends_at' => $trialEndsAt,
            'free_access_ends_at' => null,
            'license_expires_at' => null,
            'kvkk_approved_at' => now(),
            'privacy_approved_at' => now(),
            'terms_approved_at' => now(),
            'pricing_notice_approved_at' => now(),
            'consent_ip_address' => $request->ip(),
            'consent_user_agent' => substr((string) $request->userAgent(), 0, 1000),
        ]);

        Auth::login($user);

        return redirect()->route('mobile.dashboard');
    }

    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('mobile.login');
    }
}
