<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Salon;
use App\Services\SmsService;
use App\Services\Sms\OtpService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class RegisterController extends Controller
{
    protected $smsService;
    protected $otpService;

    public function __construct(SmsService $smsService, OtpService $otpService)
    {
        $this->smsService = $smsService;
        $this->otpService = $otpService;
    }
    public function showRegistrationForm()
    {
        if (Auth::check()) {
            return redirect()->route('admin.dashboard');
        }
        
        return view('auth.register');
    }

    public function checkEmail(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
        ]);

        $exists = User::where('email', $request->email)->exists();

        return response()->json([
            'exists' => $exists,
            'message' => $exists ? 'Bu e-posta adresi zaten kayıtlı.' : 'E-posta adresi kullanılabilir.',
        ]);
    }

    public function checkPhone(Request $request)
    {
        $request->validate([
            'phone' => 'required|string|min:10|max:20',
        ]);

        $phone = preg_replace('/[^0-9]/', '', $request->phone);
        if (!empty($phone) && !str_starts_with($phone, '0')) {
            $phone = '0' . $phone;
        }

        $exists = User::where('phone', $phone)->exists();

        return response()->json([
            'exists' => $exists,
            'message' => $exists ? 'Bu telefon numarası zaten kayıtlı.' : 'Telefon numarası kullanılabilir.',
        ]);
    }

    public function sendRegistrationOtp(Request $request)
    {
        \Log::info('OTP gönderme isteği alındı', [
            'request_data' => $request->all(),
            'headers' => $request->headers->all()
        ]);
        
        try {
            $validator = Validator::make($request->all(), [
                'phone' => 'required|string|min:10|max:20',
            ], [
                'phone.required' => 'Telefon numarası zorunludur.',
                'phone.min' => 'Telefon numarası en az 10 karakter olmalıdır.',
                'phone.max' => 'Telefon numarası en fazla 20 karakter olabilir.',
            ]);

            if ($validator->fails()) {
                \Log::warning('OTP validation hatası', [
                    'errors' => $validator->errors()->toArray(),
                    'request_data' => $request->all()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first('phone') ?? 'Geçersiz telefon numarası.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $phone = preg_replace('/[^0-9]/', '', $request->phone);
            if (empty($phone)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Geçerli bir telefon numarası giriniz. (05XX XXX XX XX)'
                ], 422);
            }

            if (!str_starts_with($phone, '0')) {
                $phone = '0' . $phone;
            }

            if (strlen($phone) < 10 || strlen($phone) > 11) {
                return response()->json([
                    'success' => false,
                    'message' => 'Geçerli bir telefon numarası giriniz. (05XX XXX XX XX)'
                ], 422);
            }

            if (User::where('phone', $phone)->exists()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bu telefon numarası zaten kayıtlı.',
                ], 400);
            }

            \Log::info('OTP gönderme başlatılıyor', [
                'phone' => $phone,
                'normalized_phone' => $phone,
                'request_data' => $request->all()
            ]);
            
            try {
                $result = $this->smsService->sendOtp($phone, 'registration');
            } catch (\Exception $e) {
                \Log::error('SmsService->sendOtp exception', [
                    'phone' => $phone,
                    'exception' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'SMS servisi hatası: ' . $e->getMessage(),
                    'error' => 'sms_service_exception'
                ], 500);
            }

            \Log::info('OTP gönderme sonucu', [
                'phone' => $phone,
                'success' => $result['success'] ?? false,
                'message' => $result['message'] ?? 'No message',
                'error' => $result['error'] ?? null,
                'full_result' => $result
            ]);

            if (isset($result['success']) && $result['success']) {
                return response()->json([
                    'success' => true,
                    'message' => 'Doğrulama kodu telefon numaranıza gönderildi.',
                ]);
            }

            // SmsSendResult'ta 'message' yok, 'error' var
            $errorMessage = $result['error'] ?? ($result['message'] ?? 'OTP gönderilemedi. Lütfen tekrar deneyin.');
            
            // SMS API hatası için daha kullanıcı dostu mesaj
            if (isset($result['error']) && strpos($errorMessage, '1SMS API') !== false) {
                // 1SMS API hatası detaylarını kontrol et
                if (strpos($errorMessage, 'Kod: 89') !== false) {
                    $errorMessage = 'SMS gönderim hatası: Geçersiz API isteği. Lütfen SMS ayarlarınızı kontrol edin (.env dosyasındaki SMS_1SMS_SENDER değeri geçerli olmalı).';
                } else {
                    $errorMessage = 'SMS servisi şu anda kullanılamıyor. Lütfen daha sonra tekrar deneyin veya destek ekibi ile iletişime geçin.';
                }
            }
            
            \Log::error('OTP gönderme başarısız', [
                'phone' => $phone,
                'error' => $result['error'] ?? null,
                'message' => $errorMessage,
                'result' => $result
            ]);
            
            return response()->json([
                'success' => false,
                'message' => $errorMessage,
                'error' => $result['error'] ?? null,
                'debug' => [
                    'phone' => $phone,
                    'result_success' => $result['success'] ?? null,
                    'result_error' => $result['error'] ?? null,
                ]
            ], 400);
        } catch (\Illuminate\Validation\ValidationException $e) {
            \Log::error('OTP validation hatası', [
                'errors' => $e->errors()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            \Log::error('OTP gönderme hatası: ' . $e->getMessage(), [
                'phone' => $request->phone ?? 'unknown',
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'OTP gönderilirken bir hata oluştu: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email',
            'phone' => 'required|string|min:10|max:20|unique:users,phone',
            'password' => 'required|string|min:6|confirmed',
            'otp_code' => 'required|string|size:6',
            'pricing_notice_approval' => 'accepted',
        ], [
            'name.required' => 'Ad Soyad zorunludur.',
            'email.required' => 'E-posta zorunludur.',
            'email.email' => 'Geçerli bir e-posta adresi giriniz.',
            'email.unique' => 'Bu e-posta zaten kayıtlı.',
            'phone.required' => 'Telefon numarası zorunludur.',
            'phone.min' => 'Geçerli bir telefon numarası giriniz.',
            'phone.unique' => 'Bu telefon numarası zaten kayıtlı.',
            'password.required' => 'Şifre zorunludur.',
            'password.min' => 'Şifre en az 6 karakter olmalıdır.',
            'password.confirmed' => 'Şifreler eşleşmiyor.',
            'otp_code.required' => 'Doğrulama kodu zorunludur.',
            'otp_code.size' => 'Doğrulama kodu 6 haneli olmalıdır.',
            'pricing_notice_approval.accepted' => 'Kayıt için kullanım ve ücretlendirme onayını vermeniz gerekir.',
        ]);
        
        $phone = preg_replace('/[^0-9]/', '', $request->phone);
        if (!empty($phone) && !str_starts_with($phone, '0')) {
            $phone = '0' . $phone;
        }

        $otpCode = (int) $request->otp_code;
        if (!$this->otpService->verify($phone, $otpCode, 'registration')) {
            return back()->withErrors(['otp_code' => 'Doğrulama kodu hatalı veya süresi dolmuş.'])->withInput();
        }

        try {
            DB::beginTransaction();
            $trialEndsAt = now()->addDays(14);
            
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $phone,
                'password' => Hash::make($request->password),
                'role' => 'güzellik_salonu',
                'is_active' => true,
                'sms_balance' => 0,
                'trial_ends_at' => $trialEndsAt,
                'free_access_ends_at' => null,
                'license_expires_at' => null,
                'kvkk_approved_at' => now(),
                'privacy_approved_at' => now(),
                'terms_approved_at' => now(),
                'pricing_notice_approved_at' => now(),
                'consent_ip_address' => $request->ip(),
                'consent_user_agent' => substr((string) $request->userAgent(), 0, 1000),
            ]);

            DB::commit();
            Auth::login($user);
            session()->put('show_registration_info_modal', true);

            return redirect()->route('admin.dashboard')->with('success', 'Kayıt başarılı!');
            
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Registration error: ' . $e->getMessage());
            return back()->withErrors(['error' => 'Kayıt yapılamadı.'])->withInput();
        }
    }

    public function updateRegistrationInfo(Request $request)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'salon_name' => 'required|string|max:255',
            'salon_short_name' => 'required|string|max:50',
            'salon_phone' => 'required|string',
            'salon_address' => 'required|string',
            'salon_city' => 'required|string|max:100',
            'salon_district' => 'required|string|max:100',
        ], [
            'salon_name.required' => 'Salon ismi zorunludur.',
            'salon_short_name.required' => 'Salon kısa adı zorunludur.',
            'salon_phone.required' => 'Salon telefonu zorunludur.',
            'salon_address.required' => 'Salon adresi zorunludur.',
            'salon_city.required' => 'İl seçimi zorunludur.',
            'salon_district.required' => 'İlçe seçimi zorunludur.',
        ]);

        $workingHoursValidation = $this->normalizeSalonWorkingHours($request->input('working_hours', []));
        if (!$workingHoursValidation['ok']) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $workingHoursValidation['message'],
                    'errors' => [
                        'working_hours' => [$workingHoursValidation['message']],
                    ],
                ], 422);
            }

            return back()
                ->withErrors(['working_hours' => $workingHoursValidation['message']])
                ->withInput();
        }

        $salonSlug = Str::slug($validated['salon_short_name']);
        $slugCounter = 1;
        $originalSlug = $salonSlug;
        while (User::where('salon_slug', $salonSlug)->where('id', '!=', $user->id)->exists()) {
            $salonSlug = $originalSlug . '-' . $slugCounter;
            $slugCounter++;
        }

        $salonPhone = preg_replace('/[^0-9]/', '', $validated['salon_phone']);
        if (!empty($salonPhone) && !str_starts_with($salonPhone, '0')) {
            $salonPhone = '0' . $salonPhone;
        }

        $user->salon_name = $validated['salon_name'];
        $user->salon_short_name = $validated['salon_short_name'];
        $user->salon_slug = $salonSlug;
        $user->salon_phone = $salonPhone;
        $user->salon_address = $validated['salon_address'];
        $user->salon_city = $validated['salon_city'];
        $user->salon_district = $validated['salon_district'];
        $user->salon_working_hours = $workingHoursValidation['data'];
        $user->save();

        $salon = \App\Models\Salon::where('created_by', $user->id)->first();
        if (!$salon) {
            \App\Models\Salon::create([
                'name' => $validated['salon_name'],
                'phone' => $salonPhone,
                'address' => $validated['salon_address'],
                'city' => $validated['salon_city'],
                'district' => $validated['salon_district'],
                'capacity' => 50, 
                'is_active' => true,
                'created_by' => $user->id,
            ]);
        } else {
            $salon->name = $validated['salon_name'];
            $salon->phone = $salonPhone;
            $salon->address = $validated['salon_address'];
            $salon->city = $validated['salon_city'];
            $salon->district = $validated['salon_district'];
            $salon->save();
        }

        session()->forget('show_registration_info_modal');

        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Kayıt bilgileri başarıyla güncellendi.',
            ]);
        }

        return redirect()->route('admin.dashboard')->with('success', 'Kayıt bilgileri başarıyla güncellendi.');
    }

    private function normalizeSalonWorkingHours(array $input): array
    {
        $days = [
            'monday' => 'Pazartesi',
            'tuesday' => 'Salı',
            'wednesday' => 'Çarşamba',
            'thursday' => 'Perşembe',
            'friday' => 'Cuma',
            'saturday' => 'Cumartesi',
            'sunday' => 'Pazar',
        ];

        $normalized = [];
        $openDayCount = 0;

        foreach ($days as $dayKey => $dayLabel) {
            $dayData = $input[$dayKey] ?? [];
            $isOpen = filter_var($dayData['is_open'] ?? false, FILTER_VALIDATE_BOOLEAN);

            $normalized[$dayKey] = [
                'is_open' => $isOpen,
                'open_time' => null,
                'close_time' => null,
            ];

            if (!$isOpen) {
                continue;
            }

            $openTime = trim((string) ($dayData['open_time'] ?? ''));
            $closeTime = trim((string) ($dayData['close_time'] ?? ''));

            if (!preg_match('/^\d{2}:\d{2}$/', $openTime) || !preg_match('/^\d{2}:\d{2}$/', $closeTime)) {
                return [
                    'ok' => false,
                    'message' => $dayLabel . ' için açılış ve kapanış saatini seçmelisiniz.',
                ];
            }

            if ($openTime >= $closeTime) {
                return [
                    'ok' => false,
                    'message' => $dayLabel . ' için kapanış saati açılış saatinden sonra olmalıdır.',
                ];
            }

            $normalized[$dayKey]['open_time'] = $openTime;
            $normalized[$dayKey]['close_time'] = $closeTime;
            $openDayCount++;
        }

        if ($openDayCount === 0) {
            return [
                'ok' => false,
                'message' => 'En az bir günü açık olarak seçmelisiniz.',
            ];
        }

        return [
            'ok' => true,
            'data' => $normalized,
        ];
    }

    public function storeApplication(Request $request)
    {
        $validated = $request->validate([
            'company_name' => 'required|string|max:255',
            'contact_person' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'required|string|unique:users,phone',
            'city' => 'nullable|string|max:255',
            'address' => 'nullable|string',
            'description' => 'nullable|string',
        ]);

        $phone = preg_replace('/[^0-9]/', '', $validated['phone']);
        if (!empty($phone) && !str_starts_with($phone, '0')) {
            $phone = '0' . $phone;
        }

        $fullAddress = '';
        if (!empty($validated['city'])) {
            $fullAddress .= $validated['city'];
        }
        if (!empty($validated['address'])) {
            if (!empty($fullAddress)) {
                $fullAddress .= ', ';
            }
            $fullAddress .= $validated['address'];
        }

        $user = User::create([
            'name' => $validated['contact_person'],
            'email' => $validated['email'],
            'phone' => $phone,
            'password' => Hash::make(Str::random(16)), 
            'role' => 'güzellik_salonu',
            'is_active' => false, 
            'salon_name' => $validated['company_name'],
            'salon_address' => !empty($fullAddress) ? $fullAddress : null,
            'salon_description' => $validated['description'] ?? null,
        ]);

        return redirect()->route('pages.register')
            ->with('success', 'Başvurunuz alındı! Yönetici onayından sonra hesabınız aktif olacaktır. E-posta adresinize bilgilendirme mesajı gönderilecektir.');
    }
}
