<?php

namespace App\Http\Controllers;

use App\Services\Payment\PaymentService;
use App\Models\Package;
use App\Models\User;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class PaymentController extends Controller
{
    protected $paymentService;

    public function __construct(PaymentService $paymentService)
    {
        $this->paymentService = $paymentService;
    }

    public function checkoutPackage(Request $request, $packageId)
    {
        $user = Auth::user();
        if (!$user || $user->isSuperAdmin()) {
            return redirect()->route('admin.dashboard')->with('error', 'Geçersiz işlem.');
        }

        $package = Package::findOrFail($packageId);
        return view('payment.checkout', [
            'package' => $package,
            'user' => $user,
        ]);
    }

    public function initiatePackagePayment(Request $request, $packageId)
    {
        $user = Auth::user();
        if (!$user || $user->isSuperAdmin()) {
            return redirect()->route('admin.dashboard')->with('error', 'Geçersiz işlem.');
        }

        $package = Package::findOrFail($packageId);
        return redirect()->route('payment.checkout.package', $packageId);
    }

    public function processPackagePayment(Request $request, $packageId)
    {
        $user = Auth::user();
        if (!$user || $user->isSuperAdmin()) {
            return redirect()->route('admin.dashboard')->with('error', 'Geçersiz işlem.');
        }

        $package = Package::findOrFail($packageId);
        $paymentMethod = $request->input('payment_method');

        if ($paymentMethod === 'paytr') {
            // Fatura bilgilerini kaydet
            $invoiceName = $request->input('invoice_name', $user->invoice_name ?? $user->name);
            $invoiceAddress = $request->input('invoice_address', $user->invoice_address ?? 'Türkiye');
            $invoiceCity = $request->input('invoice_city', $user->invoice_city ?? '');
            $invoiceDistrict = $request->input('invoice_district', $user->invoice_district ?? '');
            $fullAddress = trim($invoiceAddress . ($invoiceCity ? ', ' . $invoiceCity : '') . ($invoiceDistrict ? ' / ' . $invoiceDistrict : '')) ?: 'Türkiye';

            $user->update(array_filter([
                'invoice_type' => $request->input('invoice_type'),
                'invoice_name' => $invoiceName,
                'invoice_tax_number' => $request->input('invoice_tax_number'),
                'invoice_tax_office' => $request->input('invoice_tax_office'),
                'invoice_address' => $invoiceAddress,
                'invoice_city' => $invoiceCity,
                'invoice_district' => $invoiceDistrict,
            ]));

            $params = [
                'merchant_oid' => 'PKG_' . $packageId . '_' . $user->id . '_' . time(),
                'email' => $user->email,
                'payment_amount' => $package->price,
                'product_name' => $package->name . ' Paketi',
                'merchant_ok_url' => url('/odeme/basarili'),
                'merchant_fail_url' => url('/odeme/basarisiz'),
                'callback_url' => url('/odeme/callback'),
                'buyer_name' => $invoiceName,
                'buyer_phone' => $user->phone ?? '05000000000',
                'buyer_address' => $fullAddress,
            ];

            $result = $this->paymentService->getDirectPaymentFormData($params);

            if (isset($result['success']) && $result['success']) {
                session(['payment_type' => 'package', 'package_id' => $packageId, 'merchant_oid' => $params['merchant_oid']]);
                return view('payment.direct-card', [
                    'package' => $package,
                    'user' => $user,
                    'form_data' => $result,
                ]);
            }

            return redirect()->back()->with('error', 'Ödeme başlatılamadı: ' . ($result['message'] ?? 'Bilinmeyen hata'));
        } elseif ($paymentMethod === 'bank_transfer') {
            // Fatura bilgilerini kaydet
            $user->update(array_filter([
                'invoice_type' => $request->input('invoice_type'),
                'invoice_name' => $request->input('invoice_name', $user->invoice_name ?? $user->name),
                'invoice_tax_number' => $request->input('invoice_tax_number'),
                'invoice_tax_office' => $request->input('invoice_tax_office'),
                'invoice_address' => $request->input('invoice_address', $user->invoice_address ?? 'Türkiye'),
                'invoice_city' => $request->input('invoice_city'),
                'invoice_district' => $request->input('invoice_district'),
            ]));

            $merchantOid = 'PKG_' . $packageId . '_' . $user->id . '_' . time();
            
            // Transaction oluştur (pending durumunda)
            Transaction::create([
                'user_id' => $user->id,
                'type' => 'income',
                'amount' => $package->price,
                'date' => now(),
                'description' => $package->name . ' Paketi Satın Alma (Havale/EFT)',
                'status' => 'pending',
                'payment_method' => 'bank_transfer',
                'payment_reference' => $merchantOid,
            ]);

            session(['payment_type' => 'package', 'package_id' => $packageId, 'merchant_oid' => $merchantOid, 'payment_method' => 'bank_transfer']);

            return view('payment.bank-transfer', [
                'package' => $package,
                'user' => $user,
                'merchant_oid' => $merchantOid,
            ]);
        }

        return redirect()->back()->with('error', 'Geçersiz ödeme yöntemi.');
    }

    public function initiateSmsPayment(Request $request)
    {
        $user = Auth::user();
        if (!$user || $user->isSuperAdmin()) {
            return redirect()->route('admin.dashboard')->with('error', 'Geçersiz işlem.');
        }

        $smsCount = $request->input('sms_count', 100);
        $price = $request->input('price', 1000);

        $params = [
            'merchant_oid' => 'SMS_' . $user->id . '_' . time(),
            'email' => $user->email,
            'payment_amount' => $price,
            'product_name' => $smsCount . ' SMS Paketi',
            'merchant_ok_url' => url('/odeme/basarili'),
            'merchant_fail_url' => url('/odeme/basarisiz'),
            'callback_url' => url('/odeme/callback'),
            'buyer_name' => $user->name,
            'buyer_email' => $user->email,
            'buyer_phone' => $user->phone ?? '',
            'buyer_id' => (string)$user->id,
        ];

        $result = $this->paymentService->createPayment($params);

        if (isset($result['success']) && $result['success']) {
            session(['payment_type' => 'sms', 'sms_count' => $smsCount, 'merchant_oid' => $params['merchant_oid']]);
            
            if ($this->paymentService->getProvider() === 'iyzico') {
                return view('payment.iyzico', ['checkout_form' => $result['checkout_form_content'] ?? '']);
            } else {
                return view('payment.paytr', ['payment_data' => $result]);
            }
        }

        return redirect()->back()->with('error', 'Ödeme başlatılamadı: ' . ($result['message'] ?? 'Bilinmeyen hata'));
    }

    public function callback(Request $request)
    {
        $result = $this->paymentService->verifyCallback($request->all());

        if (isset($result['success']) && $result['success']) {
            $merchantOid = $result['merchant_oid'] ?? $result['conversation_id'] ?? null;
            
            if ($merchantOid && strpos($merchantOid, 'PKG_') === 0) {
                $this->processPackagePaymentCallback($merchantOid, $result);
            } elseif ($merchantOid && strpos($merchantOid, 'SMS_') === 0) {
                $this->processSmsPayment($merchantOid, $result);
            }
        }

        if ($this->paymentService->getProvider() === 'paytr') {
            echo "OK";
            exit;
        }

        return response()->json(['status' => 'ok']);
    }

    public function success(Request $request)
    {
        $paymentType = session('payment_type');
        
        if ($paymentType === 'package') {
            return redirect()->route('admin.packages.select')->with('success', 'Paket başarıyla satın alındı!');
        } elseif ($paymentType === 'sms') {
            return redirect()->route('admin.sms-packages.index')->with('success', 'SMS paketi başarıyla satın alındı!');
        }

        return redirect()->route('admin.dashboard')->with('success', 'Ödeme başarılı!');
    }

    public function fail(Request $request)
    {
        return redirect()->back()->with('error', 'Ödeme işlemi başarısız oldu.');
    }

    protected function processPackagePaymentCallback($merchantOid, $result)
    {
        $parts = explode('_', $merchantOid);
        if (count($parts) < 3) return;

        $packageId = $parts[1];
        $userId = $parts[2];
        $user = User::find($userId);
        $package = Package::find($packageId);

        if (!$user || !$package) return;

        $currentExpiry = $user->license_expires_at ? Carbon::parse($user->license_expires_at) : Carbon::now();
        if ($currentExpiry->isPast()) {
            $newExpiry = Carbon::now()->addMonths($package->duration_months ?? 1);
        } else {
            $newExpiry = $currentExpiry->copy()->addMonths($package->duration_months ?? 1);
        }

        $user->update(['license_expires_at' => $newExpiry]);

        // Transaction'ı güncelle veya oluştur
        $transaction = Transaction::where('payment_reference', $merchantOid)->first();
        if ($transaction) {
            $transaction->update([
                'status' => 'completed',
                'payment_method' => 'paytr',
            ]);
        } else {
            Transaction::create([
                'user_id' => $userId,
                'type' => 'income',
                'amount' => $result['total_amount'] ?? $result['paid_price'] ?? $package->price,
                'date' => now(),
                'description' => $package->name . ' Paketi Satın Alma',
                'status' => 'completed',
                'payment_method' => 'paytr',
                'payment_reference' => $merchantOid,
            ]);
        }

        // Kullanıcının package_id'sini güncelle
        $user->update([
            'package_id' => $packageId,
            'license_expires_at' => $newExpiry,
        ]);

        Log::info('Package payment processed', ['user_id' => $userId, 'package_id' => $packageId]);
    }

    protected function processSmsPayment($merchantOid, $result)
    {
        $parts = explode('_', $merchantOid);
        if (count($parts) < 2) return;

        $userId = $parts[1];
        $user = User::find($userId);
        if (!$user) return;

        $smsCount = session('sms_count', 100);
        $user->increment('sms_balance', $smsCount);

        $merchantOid = 'SMS_' . $userId . '_' . time();
        $transaction = Transaction::where('payment_reference', $merchantOid)->first();
        if ($transaction) {
            $transaction->update([
                'status' => 'completed',
                'payment_method' => 'paytr',
            ]);
        } else {
            Transaction::create([
                'user_id' => $userId,
                'type' => 'income',
                'amount' => $result['total_amount'] ?? $result['paid_price'] ?? 0,
                'date' => now(),
                'description' => $smsCount . ' SMS Paketi Satın Alma',
                'status' => 'completed',
                'payment_method' => 'paytr',
                'payment_reference' => $merchantOid,
            ]);
        }

        Log::info('SMS payment processed', ['user_id' => $userId, 'sms_count' => $smsCount]);
    }

    public function confirmBankTransfer(Request $request, $packageId)
    {
        $user = Auth::user();
        if (!$user || $user->isSuperAdmin()) {
            return redirect()->route('admin.dashboard')->with('error', 'Geçersiz işlem.');
        }

        $package = Package::findOrFail($packageId);
        $merchantOid = $request->input('merchant_oid');

        // Transaction'ı güncelle veya oluştur
        $transaction = Transaction::where('payment_reference', $merchantOid)
            ->where('user_id', $user->id)
            ->first();

        if ($transaction) {
            $transaction->update([
                'status' => 'pending', // Onay bekliyor
                'description' => $package->name . ' Paketi Satın Alma (Havale/EFT)' . ($request->input('notes') ? ' - ' . $request->input('notes') : ''),
            ]);
        } else {
            Transaction::create([
                'user_id' => $user->id,
                'type' => 'income',
                'amount' => $package->price,
                'date' => $request->input('payment_date', now()),
                'description' => $package->name . ' Paketi Satın Alma (Havale/EFT)' . ($request->input('notes') ? ' - ' . $request->input('notes') : ''),
                'status' => 'pending',
                'payment_method' => 'bank_transfer',
                'payment_reference' => $merchantOid,
            ]);
        }

        return redirect()->route('admin.settings.personal-invoices')
            ->with('success', 'Ödeme talebiniz alındı. Onaylandıktan sonra paketiniz aktif olacaktır.');
    }
}

