<?php

namespace App\Http\Controllers\Personel;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use Illuminate\Http\Request;

class PersonelDashboardController extends Controller
{
    public function index()
    {
        // Personel için basit dashboard - sadece rezervasyon listesi ve takvim
        return view('personel.dashboard');
    }

    public function appointments(Request $request)
    {
        $user = auth()->user();

        $query = Reservation::with(['customer', 'appointmentRequest.service', 'room'])
            ->where('personel_id', $user->id);

        // Filtreleme
        if ($request->has('status') && $request->status !== 'all') {
            if ($request->status === 'active') {
                $query->whereNotIn('status', ['cancelled', 'completed']);
            } else {
                $query->where('status', $request->status);
            }
        }

        if ($request->has('date_from') && $request->date_from) {
            $query->where('date', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to) {
            $query->where('date', '<=', $request->date_to);
        }

        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->whereHas('customer', function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        $appointments = $query->orderBy('date', 'desc')
            ->orderBy('start_time', 'desc')
            ->paginate(20);

        return view('personel.appointments', compact('appointments'));
    }
    
    public function calendar()
    {
        return view('personel.calendar');
    }
    
    public function calendarFeed(Request $request)
    {
        try {
            $start = $request->get('start');
            $end = $request->get('end');
            
            if (!$start || !$end) {
                return response()->json([]);
            }
            
            $user = auth()->user();
            
            if (!$user) {
                return response()->json([]);
            }

            $query = Reservation::with(['customer', 'appointmentRequest.service', 'room'])
                ->where('personel_id', $user->id);
            
            if ($start && $end) {
                $query->whereBetween('date', [$start, $end]);
            }
            
            $query->whereNotIn('status', ['cancelled']);
            
            $appointments = $query->get()->map(function($reservation) use ($user) {
                try {
                    if (!$reservation) {
                        return null;
                    }
                    
                    $status = $reservation->status ?? 'open';
                    $color = '#f59e0b';
                    
                    switch($status) {
                        case 'confirmed':
                            $color = '#ec4899';
                            break;
                        case 'open':
                            $color = '#f59e0b';
                            break;
                        case 'completed':
                            $color = '#22c55e';
                            break;
                        case 'cancelled':
                            $color = '#ef4444';
                            break;
                    }

                    $serviceName = 'Hizmet';
                    if ($reservation->appointmentRequest && $reservation->appointmentRequest->service) {
                        $serviceName = $reservation->appointmentRequest->service->name ?? 'Hizmet';
                    }
                    
                    $customerName = 'Müşteri';
                    if ($reservation->customer) {
                        $customerName = $reservation->customer->name ?? 'Müşteri';
                    }
                    
                    $title = $customerName . ' - ' . $serviceName;
                    
                    $date = $reservation->date;
                    if (is_string($date)) {
                        $dateStr = $date;
                    } else if (is_object($date) && method_exists($date, 'format')) {
                        $dateStr = $date->format('Y-m-d');
                    } else {
                        $dateStr = date('Y-m-d', strtotime($date));
                    }
                    
                    $startTime = $reservation->start_time ?? '00:00:00';
                    $endTime = $reservation->end_time ?? '00:00:00';
                    
                    if (strlen($startTime) > 8) {
                        $startTime = substr($startTime, 0, 8);
                    }
                    if (strlen($endTime) > 8) {
                        $endTime = substr($endTime, 0, 8);
                    }
                    
                    return [
                        'id' => $reservation->id ?? null,
                        'title' => $title,
                        'start' => $dateStr . 'T' . $startTime,
                        'end' => $dateStr . 'T' . $endTime,
                        'color' => $color,
                        'status' => $status,
                        'extendedProps' => [
                            'reservation_id' => $reservation->id ?? null,
                            'status' => $status,
                            'customer' => $customerName,
                            'personel' => $user->name ?? '',
                            'service' => $serviceName,
                            'room' => ($reservation->room && $reservation->room->name) ? $reservation->room->name : '',
                            'start_time' => $startTime,
                            'total_price' => $reservation->total_price ?? 0,
                            'notes' => $reservation->notes ?? ''
                        ]
                    ];
                } catch (\Exception $e) {
                    \Log::error('Personel calendar feed mapping hatası', [
                        'reservation_id' => $reservation->id ?? null,
                        'error' => $e->getMessage()
                    ]);
                    return null;
                }
            })->filter(function($item) {
                return $item !== null && isset($item['start']) && $item['start'] !== null;
            })->values();
            
            return response()->json($appointments);
        } catch (\Exception $e) {
            \Log::error('Personel calendar feed hatası', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([]);
        }
    }
}

