<?php

namespace App\Http\Controllers\Personel;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class ProfileController extends Controller
{
    public function edit()
    {
        $user = Auth::user();
        
        if ($user->role !== 'personel') {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }
        
        return view('personel.profile.edit', compact('user'));
    }
    
    public function update(Request $request)
    {
        $user = Auth::user();
        
        if ($user->role !== 'personel') {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'nullable|string|max:20',
            'password' => 'nullable|string|min:6',
            'profile_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:5120',
        ]);
        
        $updateData = [
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'] ?? null,
        ];
        
        if ($request->filled('password')) {
            $updateData['password'] = Hash::make($validated['password']);
        }

        if ($request->hasFile('profile_image')) {
            try {
                $file = $request->file('profile_image');
                
                if ($file->getSize() > 5120 * 1024) {
                    return back()->withErrors(['profile_image' => 'Dosya boyutu çok büyük. Maksimum 5 MB olmalıdır.'])->withInput();
                }
                
                $allowedMimes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif', 'image/webp'];
                if (!in_array($file->getMimeType(), $allowedMimes)) {
                    return back()->withErrors(['profile_image' => 'Desteklenmeyen dosya formatı.'])->withInput();
                }
                
                if ($user->profile_image && Storage::disk('public')->exists($user->profile_image)) {
                    try {
                        Storage::disk('public')->delete($user->profile_image);
                    } catch (\Exception $e) {
                        
                    }
                }
                
                $directory = 'personel-profiles';
                if (!Storage::disk('public')->exists($directory)) {
                    try {
                        Storage::disk('public')->makeDirectory($directory, 0755, true);
                    } catch (\Exception $dirException) {
                        Log::error('Dizin oluşturma hatası (personel-profiles): ' . $dirException->getMessage());
                        return back()->withErrors(['profile_image' => 'Dizin oluşturulamadı. Lütfen storage/app/public dizininin yazılabilir olduğundan emin olun.'])->withInput();
                    }
                }
                
                try {
                    $stored = $file->store($directory, 'public');
                    
                    if (!$stored) {
                        Log::error('Profil fotoğrafı store() metodu false döndü', [
                            'directory' => $directory,
                        ]);
                        return back()->withErrors(['profile_image' => 'Dosya yüklenirken bir hata oluştu.'])->withInput();
                    }
                    
                    $updateData['profile_image'] = $stored;
                } catch (\Exception $storeException) {
                    Log::error('Profil fotoğrafı yükleme hatası: ' . $storeException->getMessage());
                    return back()->withErrors(['profile_image' => 'Dosya yüklenirken bir hata oluştu: ' . $storeException->getMessage()])->withInput();
                }
            } catch (\Exception $e) {
                Log::error('Profil fotoğrafı yükleme genel hatası: ' . $e->getMessage());
                return back()->withErrors(['profile_image' => 'Dosya yüklenirken bir hata oluştu.'])->withInput();
            }
        }
        
        try {
            $user->update($updateData);
            
            return redirect()->route('personel.profile.edit')->with('success', 'Profil bilgileriniz başarıyla güncellendi.');
        } catch (\Exception $e) {
            Log::error('Profil güncelleme hatası: ' . $e->getMessage());
            return back()->withErrors(['error' => 'Profil güncellenirken bir hata oluştu.'])->withInput();
        }
    }
}

