<?php

namespace App\Http\Controllers\Public;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\AppointmentRequest;
use App\Models\User;
use App\Models\Service;
use App\Models\Reservation;
use App\Models\PersonelLeave;
use Carbon\Carbon;

class AppointmentRequestController extends Controller
{
    public function sendOTP(Request $request)
    {
        $validated = $request->validate([
            'phone' => 'required|string|min:10',
            'salon_id' => 'required|integer|exists:users,id',
        ]);

        try {
            $salon = \App\Models\User::find($validated['salon_id']);
            if (!$salon) {
                return response()->json([
                    'success' => false,
                    'message' => 'Salon bulunamadı.',
                ], 404);
            }

            $phone = preg_replace('/[^0-9]/', '', $validated['phone']);
            if (!empty($phone) && !str_starts_with($phone, '0')) {
                $phone = '0' . $phone;
            }

            $customer = \App\Models\Customer::where('phone', $phone)
                ->where('beauty_salon_id', $salon->id)
                ->first();
            
            $isRegistered = false;
            $customerData = null;
            
            if ($customer) {
                $isRegistered = true;

                $nameParts = explode(' ', trim($customer->name), 2);
                $customerName = $nameParts[0];
                $customerSurname = count($nameParts) > 1 ? $nameParts[1] : '';

                $lastAppointmentRequest = \App\Models\AppointmentRequest::where('customer_id', $customer->id)
                    ->where('beauty_salon_id', $salon->id)
                    ->whereNotNull('age')
                    ->orderBy('created_at', 'desc')
                    ->first();
                
                $customerAge = $lastAppointmentRequest ? $lastAppointmentRequest->age : null;
                
                $customerData = [
                    'customer_id' => $customer->id,
                    'customer_name' => $customerName,
                    'customer_surname' => $customerSurname,
                    'customer_age' => $customerAge,
                ];
            }
            
            $smsService = app(\App\Services\SmsService::class);

            if (!$salon->isSuperAdmin() && !$salon->hasSmsBalance(1)) {
                return response()->json([
                    'success' => false,
                    'message' => 'SMS bakiyeniz yetersiz. Lütfen SMS paketi satın alın.',
                ], 400);
            }
            
            $result = $smsService->sendOtp($phone, 'appointment_request');

            if ($result['success']) {
                
                if (!$salon->isSuperAdmin()) {
                    $salon->deductSmsBalance(1);
                }
                
                $response = [
                    'success' => true,
                    'message' => 'Doğrulama kodu telefon numaranıza gönderildi.',
                    'is_registered' => $isRegistered,
                ];

                if ($isRegistered && $customerData) {
                    $response = array_merge($response, $customerData);
                }
                
                return response()->json($response);
            }

            return response()->json([
                'success' => false,
                'message' => $result['message'] ?? 'OTP gönderilemedi. Lütfen tekrar deneyin.',
            ], 400);
        } catch (\Exception $e) {
            \Log::error('Salon OTP gönderme hatası', [
                'phone' => $validated['phone'],
                'salon_id' => $validated['salon_id'],
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'OTP gönderilirken bir hata oluştu. Lütfen tekrar deneyin.',
            ], 500);
        }
    }

    public function verifyOTP(Request $request)
    {
        $validated = $request->validate([
            'phone' => 'required|string',
            'otp_code' => 'required|string',
            'otp' => 'nullable|string', 
        ]);

        $otpCode = $validated['otp_code'] ?? $validated['otp'] ?? null;
        
        if (!$otpCode) {
            return response()->json([
                'success' => false,
                'message' => 'OTP kodu gerekli.',
            ], 400);
        }

        try {
            
            $phone = preg_replace('/[^0-9]/', '', $validated['phone']);
            if (!empty($phone) && !str_starts_with($phone, '0')) {
                $phone = '0' . $phone;
            }

            $smsService = app(\App\Services\SmsService::class);
            $isValid = $smsService->verifyOtp($phone, (int)$otpCode, 'appointment_request');

            if ($isValid) {
                return response()->json([
                    'success' => true,
                    'message' => 'OTP doğrulandı.',
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'OTP kodu hatalı veya süresi dolmuş.',
            ], 400);
        } catch (\Exception $e) {
            \Log::error('OTP doğrulama hatası', [
                'phone' => $validated['phone'],
                'otp_code' => $otpCode,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'OTP doğrulanırken bir hata oluştu. Lütfen tekrar deneyin.',
            ], 500);
        }
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'salon_id' => 'required|exists:users,id',
            'phone' => 'required|string',
            'name' => 'required|string|max:255',
            'service_id' => 'nullable',
            'service_ids' => 'nullable|array',
            'service_ids.*' => 'nullable|exists:services,id',
            'personel_id' => 'nullable|exists:users,id',
            'date' => 'nullable|date',
            'time' => 'nullable|string',
            'surname' => 'nullable|string|max:255',
            'age' => 'nullable|integer|min:1|max:150',
            'notes' => 'nullable|string',
            'referral_code' => 'nullable|string|size:6',
            'recurring' => 'nullable|array',
            'recurring.recurring_type' => 'nullable|in:weekly,monthly',
            'recurring.day_of_week' => 'nullable|integer|min:1|max:7',
            'recurring.day_of_month' => 'nullable|integer|min:1|max:31',
            'recurring.recurring_count' => 'nullable|integer|min:1|max:24',
        ]);

        $serviceId = null;
        $serviceIds = null;
        
        if ($request->filled('service_ids') && is_array($request->service_ids) && count($request->service_ids) > 0) {
            $serviceIds = $request->service_ids;
            $serviceId = $serviceIds[0]; 
        } elseif ($request->filled('service_id')) {
            $serviceId = $validated['service_id'];
            $serviceIds = [$serviceId];
        }

        $appointmentRequest = AppointmentRequest::create([
            'beauty_salon_id' => $validated['salon_id'],
            'phone' => $validated['phone'],
            'name' => $validated['name'],
            'surname' => $validated['surname'] ?? null,
            'age' => $validated['age'] ?? null,
            'service_id' => $serviceId,
            'service_ids' => $serviceIds, 
            'personel_id' => $validated['personel_id'] ?? null,
            'appointment_date' => $validated['date'] ?? null,
            'appointment_time' => $validated['time'] ?? null,
            'notes' => $validated['notes'] ?? null,
            'referral_code' => $validated['referral_code'] ?? null,
            'status' => 'pending',
            'phone_verified' => true, 
            'recurring_data' => $validated['recurring'] ?? null, 
        ]);

        // Admin panele bildirim gönder
        $salon = \App\Models\User::find($validated['salon_id']);
        if ($salon) {
            $salon->notify(new \App\Notifications\AppointmentRequestCreated($appointmentRequest));
        }

        return response()->json([
            'success' => true,
            'message' => 'Randevu talebiniz alındı.' . (isset($validated['recurring']) ? ' Tekrarlanan seans ayarları kaydedildi.' : ''),
            'data' => $appointmentRequest,
        ]);
    }

    public function getAvailableSlots(Request $request)
    {
        try {
            
            $serviceIds = [];
            
            if ($request->has('service_ids')) {
                
                $serviceIdsParam = $request->input('service_ids');
                if (is_string($serviceIdsParam)) {
                    $serviceIds = array_filter(array_map('intval', explode(',', $serviceIdsParam)));
                } elseif (is_array($serviceIdsParam)) {
                    $serviceIds = array_filter(array_map('intval', $serviceIdsParam));
                }
            } elseif ($request->has('service_id')) {
                
                $serviceIds = [(int)$request->input('service_id')];
            }
            
            if (empty($serviceIds)) {
                return response()->json([
                    'success' => false,
                    'message' => 'En az bir hizmet seçilmelidir.',
                    'errors' => ['service_id' => ['En az bir hizmet seçilmelidir.']]
                ], 422);
            }
            
            $validated = $request->validate([
                'salon_id' => 'required|exists:users,id',
                'date' => 'nullable|date', 
            ]);

            $services = Service::whereIn('id', $serviceIds)
                ->where('beauty_salon_id', $validated['salon_id'])
                ->get();
                
            if ($services->count() !== count($serviceIds)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Seçilen hizmetlerden biri bulunamadı.',
                    'errors' => ['service_ids' => ['Seçilen hizmetlerden biri bulunamadı.']]
                ], 422);
            }

            $personelsQuery = User::where('beauty_salon_id', $validated['salon_id'])
                ->where('role', 'personel');

            $personels = $personelsQuery->whereHas('services', function($query) use ($serviceIds) {
                $query->whereIn('services.id', $serviceIds);
            })->get();

            if ($personels->isEmpty()) {
                return response()->json([
                    'success' => true,
                    'personels' => [],
                    'available_slots' => [],
                    'service_duration' => 30,
                    'message' => 'Bu hizmet(ler) için uygun personel bulunamadı.',
                ]);
            }

            $totalDuration = (int)$services->sum('duration');
            if ($totalDuration <= 0) {
                $totalDuration = 30; 
            }

            $availableSlots = [];
            $busySlots = [];

            if ($request->filled('date')) {
                $selectedDate = Carbon::parse($validated['date'])->format('Y-m-d');

                $allSlots = [];
                $startHour = 8;
                $endHour = 22;
                for ($hour = $startHour; $hour < $endHour; $hour++) {
                    $allSlots[] = str_pad($hour, 2, '0', STR_PAD_LEFT) . ':00';
                    $allSlots[] = str_pad($hour, 2, '0', STR_PAD_LEFT) . ':30';
                }
                $allSlots[] = '22:00';

                // Tüm personellerin randevularını al ve dolu saatleri belirle
                $busyTimeSlots = []; // [personel_id => [time_slots...]]
                
                foreach ($personels as $personel) {
                    $reservations = Reservation::where('beauty_salon_id', $validated['salon_id'])
                        ->where('personel_id', $personel->id)
                        ->where('date', $selectedDate)
                        ->whereNotIn('status', ['cancelled'])
                        ->with(['appointmentRequest.service'])
                        ->get();

                    $personelBusySlots = [];
                    foreach ($reservations as $reservation) {
                        if ($reservation->start_time) {
                            // Başlangıç saatini dolu yap
                            $personelBusySlots[] = $reservation->start_time;
                            
                            // Süreyi hesapla
                            $reservationDuration = 30; // Varsayılan
                            
                            // Eğer end_time varsa, o zamana kadar olan tüm slotları dolu yap
                            if ($reservation->end_time) {
                                $startTime = Carbon::createFromFormat('H:i', $reservation->start_time);
                                $endTime = Carbon::createFromFormat('H:i', $reservation->end_time);
                                
                                $currentTime = $startTime->copy()->addMinutes(30); // İlk 30 dakika zaten başlangıç slotunda
                                while ($currentTime->lt($endTime)) {
                                    $personelBusySlots[] = $currentTime->format('H:i');
                                    $currentTime->addMinutes(30);
                                }
                            } else {
                                // End_time yoksa, appointmentRequest'ten service bilgisini al ve süreyi hesapla
                                if ($reservation->appointmentRequest) {
                                    $appointmentRequest = $reservation->appointmentRequest;
                                    
                                    // service_ids array'inden service bilgilerini al
                                    if ($appointmentRequest->service_ids && is_array($appointmentRequest->service_ids) && !empty($appointmentRequest->service_ids)) {
                                        $serviceModels = Service::whereIn('id', $appointmentRequest->service_ids)->get();
                                        $reservationDuration = $serviceModels->sum('duration') ?? 30;
                                    } elseif ($appointmentRequest->service) {
                                        // Tek hizmet varsa
                                        $reservationDuration = $appointmentRequest->service->duration ?? 30;
                                    }
                                }
                                
                                if ($reservationDuration <= 0) {
                                    $reservationDuration = 30;
                                }
                                
                                $startTime = Carbon::createFromFormat('H:i', $reservation->start_time);
                                $currentTime = $startTime->copy()->addMinutes(30);
                                $endTime = $startTime->copy()->addMinutes($reservationDuration);
                                
                                while ($currentTime->lt($endTime)) {
                                    $personelBusySlots[] = $currentTime->format('H:i');
                                    $currentTime->addMinutes(30);
                                }
                            }
                        }
                    }
                    
                    $busyTimeSlots[$personel->id] = array_unique($personelBusySlots);
                }

                $leavePersonelIds = PersonelLeave::where('beauty_salon_id', $validated['salon_id'])
                    ->whereDate('leave_date', $selectedDate)
                    ->pluck('personel_id')
                    ->all();

                foreach ($leavePersonelIds as $leavePersonelId) {
                    $busyTimeSlots[$leavePersonelId] = $allSlots;
                }

                foreach ($allSlots as $slot) {
                    $busyCount = 0;
                    $availablePersonels = [];

                    // Her personel için bu saatte dolu mu kontrol et
                    foreach ($personels as $personel) {
                        $personelBusySlots = $busyTimeSlots[$personel->id] ?? [];
                        if (in_array($slot, $personelBusySlots)) {
                            $busyCount++;
                        } else {
                            $availablePersonels[] = [
                                'id' => $personel->id,
                                'name' => $personel->name,
                            ];
                        }
                    }

                    $totalPersonels = $personels->count();
                    $isAvailable = $totalPersonels > 0 && !empty($availablePersonels);

                    $availableSlots[] = [
                        'time' => $slot,
                        'available' => $isAvailable,
                        'busy_count' => $busyCount,
                        'total_personels' => $totalPersonels,
                        'available_personels' => $availablePersonels,
                    ];
                }
            }

            return response()->json([
                'success' => true,
                'personels' => $personels->map(function($personel) {
                    return [
                        'id' => $personel->id,
                        'name' => $personel->name,
                    ];
                }),
                'available_slots' => $availableSlots,
                'service_duration' => $totalDuration,
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            
            return response()->json([
                'success' => false,
                'message' => 'Geçersiz veri gönderildi.',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            \Log::error('Müsait saatler getirilirken hata', [
                'service_ids' => $serviceIds ?? null,
                'salon_id' => $request->input('salon_id'),
                'date' => $request->input('date'),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Müsait saatler getirilirken bir hata oluştu: ' . $e->getMessage(),
            ], 500);
        }
    }
}
