<?php

namespace App\Http\Controllers\Public;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Reservation;
use Illuminate\Http\Request;

class SalonPageController extends Controller
{
    public function show($slug)
    {
        
        $shortUrlSalon = User::where('short_url', $slug)
            ->where('role', 'güzellik_salonu')
            ->first();
        
        // short_url ile yakalandıysa canonical salon_slug'e yönlendir.
        // Ancak hedef slug mevcut slug ile aynıysa self-redirect döngüsü oluşur.
        if (
            $shortUrlSalon &&
            !empty($shortUrlSalon->salon_slug) &&
            trim((string) $shortUrlSalon->salon_slug) !== trim((string) $slug)
        ) {
            
            return redirect()->route('public.salon.show', ['slug' => $shortUrlSalon->salon_slug], 301);
        }

        $salon = User::where('salon_slug', $slug)
            ->where('role', 'güzellik_salonu')
            ->firstOrFail();

        $requestedTheme = $salon->salon_theme ?? 'theme1';
        
        $validThemes = ['theme1', 'theme2', 'theme3', 'theme4', 'theme5'];
        if (!in_array($requestedTheme, $validThemes)) {
            $requestedTheme = 'theme1';
        }

        // Tum temalarda yerlesim theme1 ile ayni calissin.
        $theme = 'theme1';
        
        $upcomingAppointments = Reservation::where('beauty_salon_id', $salon->id)
            ->where('date', '>=', now()->toDateString())
            ->where('status', '!=', 'cancelled')
            ->with(['customer', 'personel', 'room'])
            ->orderBy('date')
            ->orderBy('start_time')
            ->limit(10)
            ->get();

        $personels = User::where('beauty_salon_id', $salon->id)
            ->where('role', 'personel')
            ->with('services')
            ->orderBy('name')
            ->get();

        $services = \App\Models\Service::where('beauty_salon_id', $salon->id)
            ->with('category')
            ->orderBy('name')
            ->get();

        $serviceCategories = \App\Models\ServiceCategory::where('beauty_salon_id', $salon->id)
            ->withCount('services')
            ->orderBy('sort_order')
            ->orderBy('name')
            ->get();

        $averageRating = \App\Models\SalonRating::getAverageRating($salon->id);
        $ratingCount = \App\Models\SalonRating::getRatingCount($salon->id);
        $ratings = \App\Models\SalonRating::where('salon_id', $salon->id)
            ->where('is_used', true)
            ->where('rating', '>', 0)
            ->orderBy('created_at', 'desc')
            ->limit(20)
            ->get();

        $viewPath = "public.salon.{$theme}";
        
        $viewFile = resource_path("views/public/salon/{$theme}.blade.php");
        if (!file_exists($viewFile)) {
            \Log::warning("Tema dosyası bulunamadı: {$theme}, varsayılan tema kullanılıyor: theme1");
            $theme = 'theme1';
            $viewPath = "public.salon.theme1";
        }

        $colorScheme = $salon->salon_color ?? 'pink';
        $validColors = ['pink', 'blue', 'purple', 'orange', 'green'];
        if (!in_array($colorScheme, $validColors)) {
            $colorScheme = 'pink';
        }

        try {
            return view($viewPath, compact('salon', 'upcomingAppointments', 'personels', 'services', 'serviceCategories', 'theme', 'colorScheme', 'averageRating', 'ratingCount', 'ratings', 'requestedTheme'));
        } catch (\Exception $e) {
            \Log::error("Tema render hatası: " . $e->getMessage());
            return view('public.salon.theme1', compact('salon', 'upcomingAppointments', 'personels', 'services', 'serviceCategories', 'theme', 'colorScheme', 'averageRating', 'ratingCount', 'ratings', 'requestedTheme'));
        }
    }
}
