<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\Setting;
use App\Models\Contract;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;

class PublicInvoiceController extends \App\Http\Controllers\Controller
{
    public function show($token)
    {
        $invoice = Invoice::where('public_token', $token)
            ->with(['reservation.customer', 'reservation.salon', 'customer'])
            ->firstOrFail();

        // Ayarları al
        $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
        $generalSettings = Setting::where('group', 'general')->pluck('value', 'key');
        
        $logo = $settings['invoice_logo'] ?? null;
        $companyName = $generalSettings['company_name'] ?? 'Salonay';
        $companyPhone = $generalSettings['company_phone'] ?? null;
        $companyEmail = $generalSettings['company_email'] ?? null;
        
        // Sözleşme URL'i varsa al
        $contractUrl = null;
        if ($invoice->reservation && $invoice->reservation->beauty_salon_id) {
            $contract = Contract::where('user_id', $invoice->reservation->beauty_salon_id)
                ->where('is_default', true)
                ->where('is_active', true)
                ->first();
            
            if ($contract) {
                $contractUrl = $contract->public_url;
            }
        }
        
        // Transaction bilgisi varsa al (eğer invoice items_json içinde transaction_id varsa)
        $transaction = null;
        if ($invoice->items_json && is_array($invoice->items_json)) {
            $transactionId = $invoice->items_json['transaction_id'] ?? null;
            if ($transactionId) {
                $transaction = \App\Models\Transaction::find($transactionId);
            }
        }
        
        return view('public.invoice', compact('invoice', 'logo', 'companyName', 'companyPhone', 'companyEmail', 'contractUrl', 'transaction'));
    }

    public function downloadPdf($token)
    {
        $invoice = Invoice::where('public_token', $token)
            ->with(['reservation.customer', 'reservation.salon', 'customer'])
            ->firstOrFail();

        // Ayarları al
        $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
        $generalSettings = Setting::where('group', 'general')->pluck('value', 'key');
        
        $logo = $settings['invoice_logo'] ?? null;
        $address = $settings['invoice_address'] ?? '';
        $taxOffice = $settings['invoice_tax_office'] ?? '';
        $taxNumber = $settings['invoice_tax_number'] ?? '';
        $companyName = $generalSettings['company_name'] ?? 'Salonay';

        // Invoice data hazırla
        if (!$invoice->reservation) {
            // Transaction invoice için
            $invoiceData = $this->prepareTransactionInvoiceData($invoice, $logo, $address, $taxOffice, $taxNumber, $companyName);
        } else {
            // Reservation invoice için
            $reservation = $invoice->reservation;
            $salonAddress = $reservation->salon->address ?? $address;
            $invoiceData = $this->prepareInvoiceData($reservation, $logo, $salonAddress, $taxOffice, $taxNumber, $invoice);
        }

        $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);
        
        return $pdf->download('fatura-' . $invoice->invoice_number . '.pdf');
    }

    private function prepareInvoiceData($reservation, $logo, $address, $taxOffice, $taxNumber, $invoice = null)
    {
        $items = [];
        $itemNumber = 0;

        // Hizmet bilgisini bul
        $serviceName = 'Hizmet';
        $reservation->load('appointmentRequest.service');
        
        if ($reservation->appointmentRequest && $reservation->appointmentRequest->service) {
            $serviceName = $reservation->appointmentRequest->service->name;
        } else {
            $itemsDataCheck = json_decode($reservation->items_data ?? '{}', true);
            if (isset($itemsDataCheck['service']) && is_array($itemsDataCheck['service'])) {
                $serviceName = $itemsDataCheck['service']['name'] ?? 'Hizmet';
            } elseif (isset($itemsDataCheck['services']) && is_array($itemsDataCheck['services']) && !empty($itemsDataCheck['services'])) {
                $firstService = $itemsDataCheck['services'][0];
                $serviceName = $firstService['name'] ?? 'Hizmet';
            }
        }

        $itemsData = json_decode($reservation->items_data ?? '{}', true);
        if (!is_array($itemsData) || empty($itemsData)) {
            $itemsData = json_decode($reservation->notes ?? '{}', true);
            if (!is_array($itemsData)) {
                $itemsData = [];
            }
        }

        $mealTotal = 0;
        $organizationPrice = $itemsData['organization_price'] ?? 0;
        $extraTotal = 0;

        if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
            foreach ($itemsData['meals'] as $meal) {
                if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                    $total = $meal['quantity'] * $meal['price'];
                    $mealTotal += $total;
                }
            }
        }

        if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
            foreach ($itemsData['extras'] as $extra) {
                if (isset($extra['name']) && isset($extra['price']) && $extra['price'] > 0) {
                    $extraTotal += $extra['price'];
                }
            }
        }

        $salonPrice = $reservation->total_price - ($mealTotal + $organizationPrice + $extraTotal);

        // Hizmet kalemini ekle (toplam fiyat - yemek - organizasyon - extra)
        if ($salonPrice > 0) {
            $items[] = [
                'name' => $serviceName,
                'quantity' => 1,
                'price' => $salonPrice,
                'total' => $salonPrice
            ];
        }

        if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
            foreach ($itemsData['meals'] as $meal) {
                if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                    $total = $meal['quantity'] * $meal['price'];
                    $items[] = [
                        'name' => $meal['name'],
                        'quantity' => $meal['quantity'] . ' Kisi',
                        'price' => $meal['price'],
                        'total' => $total
                    ];
                }
            }
        }

        if ($organizationPrice > 0) {
            $items[] = [
                'name' => 'Organizasyon',
                'quantity' => '-',
                'price' => $organizationPrice,
                'total' => $organizationPrice
            ];
        }

        if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
            foreach ($itemsData['extras'] as $extra) {
                if (isset($extra['name']) && isset($extra['price']) && $extra['price'] > 0) {
                    $items[] = [
                        'name' => $extra['name'],
                        'quantity' => '-',
                        'price' => $extra['price'],
                        'total' => $extra['price']
                    ];
                }
            }
        }

        $grandTotal = $reservation->total_price;
        $depositPaid = $reservation->deposit_paid ?? 0;

        // Salon bilgilerini al
        $salonName = 'Güzellik Salonu';
        $salonAddress = $address;
        
        if ($reservation->beauty_salon_id) {
            $beautySalon = \App\Models\User::find($reservation->beauty_salon_id);
            if ($beautySalon) {
                $salonName = $beautySalon->salon_name ?? $beautySalon->name ?? 'Güzellik Salonu';
                $salonAddress = $beautySalon->salon_address ?? $beautySalon->address ?? $address;
            }
        } elseif ($reservation->salon) {
            $salonName = $reservation->salon->name ?? 'Güzellik Salonu';
            $salonAddress = $reservation->salon->address ?? $address;
        }
        
        return [
            'reservation' => $reservation,
            'invoice' => $invoice,
            'items' => $items,
            'subtotal' => $grandTotal,
            'grandTotal' => $grandTotal,
            'depositPaid' => $depositPaid,
            'logo' => $logo,
            'address' => $address,
            'salonName' => $salonName,
            'salonAddress' => $salonAddress,
            'taxOffice' => $taxOffice,
            'taxNumber' => $taxNumber,
        ];
    }

    private function prepareTransactionInvoiceData($invoice, $logo, $address, $taxOffice, $taxNumber, $companyName)
    {
        $items = $invoice->items_json ?? [];

        $cleanItems = [];
        foreach ($items as $item) {
            if (is_array($item)) {
                $cleanItems[] = [
                    'name' => $item['name'] ?? 'İşlem',
                    'quantity' => $item['quantity'] ?? '-',
                    'price' => $item['price'] ?? 0,
                    'total' => $item['total'] ?? ($item['price'] ?? 0),
                ];
            }
        }

        if (empty($cleanItems)) {
            $cleanItems[] = [
                'name' => 'İşlem',
                'quantity' => '-',
                'price' => $invoice->total_amount,
                'total' => $invoice->total_amount,
            ];
        }

        $customerName = $invoice->customer ? $invoice->customer->name : 'Müşteri';

        $invoiceDate = $invoice->invoice_date ?? $invoice->created_at;
        
        return [
            'reservation' => null,
            'transaction' => true,
            'invoice' => $invoice,
            'items' => $cleanItems,
            'subtotal' => $invoice->total_amount,
            'grandTotal' => $invoice->total_amount,
            'depositPaid' => $invoice->deposit_paid ?? 0,
            'logo' => $logo,
            'address' => $address,
            'salonName' => $companyName,
            'salonAddress' => $address,
            'taxOffice' => $taxOffice,
            'taxNumber' => $taxNumber,
            'customerName' => $customerName,
            'invoiceDate' => $invoiceDate,
        ];
    }
}

