<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Coupon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CouponController extends Controller
{
    public function index()
    {
        if (!Auth::user()->isSuperAdmin()) {
            abort(403);
        }
        $coupons = Coupon::orderBy('created_at', 'desc')->paginate(20);
        return view('superadmin.coupons.index', compact('coupons'));
    }

    public function create()
    {
        if (!Auth::user()->isSuperAdmin()) {
            abort(403);
        }
        return view('superadmin.coupons.create');
    }

    public function store(Request $request)
    {
        if (!Auth::user()->isSuperAdmin()) {
            abort(403);
        }
        $validated = $request->validate([
            'code' => 'required|string|max:64|unique:coupons,code',
            'name' => 'nullable|string|max:255',
            'type' => 'required|in:subscription,sms,both',
            'subscription_days' => 'nullable|integer|min:0',
            'sms_credits' => 'nullable|integer|min:0',
            'max_uses' => 'nullable|integer|min:1',
            'valid_from' => 'nullable|date',
            'valid_until' => 'nullable|date|after_or_equal:valid_from',
            'is_active' => 'boolean',
        ], [
            'code.required' => 'Kupon kodu zorunludur.',
            'code.unique' => 'Bu kupon kodu zaten kayıtlı.',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);
        if ($validated['type'] === 'subscription') {
            $validated['subscription_days'] = $validated['subscription_days'] ?? 0;
            $validated['sms_credits'] = null;
        } elseif ($validated['type'] === 'sms') {
            $validated['sms_credits'] = $validated['sms_credits'] ?? 0;
            $validated['subscription_days'] = null;
        } else {
            $validated['subscription_days'] = $validated['subscription_days'] ?? 0;
            $validated['sms_credits'] = $validated['sms_credits'] ?? 0;
        }
        $validated['code'] = strtoupper(trim($validated['code']));

        Coupon::create($validated);
        return redirect()->route('admin.superadmin.coupons.index')->with('success', 'Kupon oluşturuldu.');
    }

    public function edit($id)
    {
        if (!Auth::user()->isSuperAdmin()) {
            abort(403);
        }
        $coupon = Coupon::findOrFail($id);
        return view('superadmin.coupons.edit', compact('coupon'));
    }

    public function update(Request $request, $id)
    {
        if (!Auth::user()->isSuperAdmin()) {
            abort(403);
        }
        $coupon = Coupon::findOrFail($id);
        $validated = $request->validate([
            'code' => 'required|string|max:64|unique:coupons,code,' . $id,
            'name' => 'nullable|string|max:255',
            'type' => 'required|in:subscription,sms,both',
            'subscription_days' => 'nullable|integer|min:0',
            'sms_credits' => 'nullable|integer|min:0',
            'max_uses' => 'nullable|integer|min:1',
            'valid_from' => 'nullable|date',
            'valid_until' => 'nullable|date',
            'is_active' => 'boolean',
        ]);
        $validated['is_active'] = $request->boolean('is_active');
        if ($validated['type'] === 'subscription') {
            $validated['subscription_days'] = $validated['subscription_days'] ?? 0;
            $validated['sms_credits'] = null;
        } elseif ($validated['type'] === 'sms') {
            $validated['sms_credits'] = $validated['sms_credits'] ?? 0;
            $validated['subscription_days'] = null;
        } else {
            $validated['subscription_days'] = $validated['subscription_days'] ?? 0;
            $validated['sms_credits'] = $validated['sms_credits'] ?? 0;
        }
        $validated['code'] = strtoupper(trim($validated['code']));
        $coupon->update($validated);
        return redirect()->route('admin.superadmin.coupons.index')->with('success', 'Kupon güncellendi.');
    }

    public function destroy($id)
    {
        if (!Auth::user()->isSuperAdmin()) {
            abort(403);
        }
        $coupon = Coupon::findOrFail($id);
        $coupon->delete();
        return redirect()->route('admin.superadmin.coupons.index')->with('success', 'Kupon silindi.');
    }
}
