<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Contract;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class GlobalContractController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }
        
        $globalContract = Contract::getGlobalContract();
        
        return view('admin.superadmin.global-contract', compact('globalContract'));
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403, 'Bu işlemi yapma yetkiniz yok.');
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'content' => 'required|string',
        ]);
        
        $existingGlobal = Contract::getGlobalContract();
        
        if ($existingGlobal) {
            $existingGlobal->update([
                'name' => $validated['name'],
                'content' => $validated['content'],
                'is_active' => true,
                'is_global' => true,
                'is_required' => true,
            ]);
            
            $contract = $existingGlobal;
        } else {
            $contract = Contract::create([
                'name' => $validated['name'],
                'content' => $validated['content'],
                'is_active' => true,
                'is_global' => true,
                'is_required' => true,
                'is_default' => false,
                'user_id' => $user->id,
                'company_id' => null,
            ]);
        }
        
        Log::info('Global contract updated', [
            'user_id' => $user->id,
            'contract_id' => $contract->id,
            'contract_name' => $contract->name
        ]);
        
        return redirect()->route('admin.superadmin.global-contract.index')
            ->with('success', 'Ana sözleşme başarıyla kaydedildi.');
    }

    public function destroy(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403, 'Bu işlemi yapma yetkiniz yok.');
        }
        
        $globalContract = Contract::getGlobalContract();
        
        if ($globalContract) {
            $contractId = $globalContract->id;
            $contractName = $globalContract->name;
            
            $globalContract->update([
                'is_active' => false,
                'is_global' => false,
                'is_required' => false,
            ]);
            
            Log::info('Global contract deactivated', [
                'user_id' => $user->id,
                'contract_id' => $contractId,
                'contract_name' => $contractName
            ]);
            
            return redirect()->route('admin.superadmin.global-contract.index')
                ->with('success', 'Ana sözleşme başarıyla silindi.');
        }
        
        return redirect()->route('admin.superadmin.global-contract.index')
            ->with('error', 'Ana sözleşme bulunamadı.');
    }
}

