<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Package;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class PackageController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $packages = Package::orderBy('sort_order')->orderBy('name')->get();
        
        return view('superadmin.packages.index', compact('packages'));
    }

    public function create()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        return view('superadmin.packages.create');
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'price_period' => 'required|in:monthly,yearly',
            'duration_months' => 'required|integer|min:1',
            'is_popular' => 'boolean',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer',
            'features' => 'nullable|array',
            'features.*' => 'string',
        ]);

        $package = Package::create([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'price' => $validated['price'],
            'price_period' => $validated['price_period'],
            'duration_months' => $validated['duration_months'],
            'is_popular' => $request->has('is_popular') && $request->is_popular == '1',
            'is_active' => $request->has('is_active') ? ($request->is_active == '1') : true,
            'sort_order' => $validated['sort_order'] ?? 0,
            'features' => $validated['features'] ?? [],
        ]);

        return redirect()->route('admin.superadmin.packages.index')
            ->with('success', 'Paket başarıyla oluşturuldu.');
    }

    public function edit($id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $package = Package::findOrFail($id);
        
        return view('superadmin.packages.edit', compact('package'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $package = Package::findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'price_period' => 'required|in:monthly,yearly',
            'duration_months' => 'required|integer|min:1',
            'is_popular' => 'boolean',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer',
            'features' => 'nullable|array',
            'features.*' => 'string',
        ]);

        $package->update([
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'price' => $validated['price'],
            'price_period' => $validated['price_period'],
            'duration_months' => $validated['duration_months'],
            'is_popular' => $request->has('is_popular') && $request->is_popular == '1',
            'is_active' => $request->has('is_active') ? ($request->is_active == '1') : true,
            'sort_order' => $validated['sort_order'] ?? 0,
            'features' => $validated['features'] ?? [],
        ]);

        return redirect()->route('admin.superadmin.packages.index')
            ->with('success', 'Paket başarıyla güncellendi.');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $package = Package::findOrFail($id);
        $package->delete();

        return redirect()->route('admin.superadmin.packages.index')
            ->with('success', 'Paket başarıyla silindi.');
    }
}
