<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\User;
use App\Models\Package;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class PendingPaymentsController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $pendingPayments = Transaction::where('status', 'pending')
            ->where('payment_method', 'bank_transfer')
            ->where('type', 'income')
            ->whereNotNull('payment_reference')
            ->with(['user', 'user.package'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);
        
        return view('superadmin.pending-payments.index', compact('pendingPayments'));
    }

    public function approve(Request $request, $id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $transaction = Transaction::findOrFail($id);
        
        if ($transaction->status !== 'pending') {
            return redirect()->back()->with('error', 'Bu ödeme zaten işlenmiş.');
        }
        
        $merchantOid = $transaction->payment_reference;
        
        // Paket ödemesi mi kontrol et
        if (strpos($merchantOid, 'PKG_') === 0) {
            $parts = explode('_', $merchantOid);
            if (count($parts) >= 3) {
                $packageId = $parts[1];
                $userId = $parts[2];
                $targetUser = User::find($userId);
                $package = Package::find($packageId);
                
                if ($targetUser && $package) {
                    // Lisans süresini uzat
                    $currentExpiry = $targetUser->license_expires_at ? Carbon::parse($targetUser->license_expires_at) : Carbon::now();
                    if ($currentExpiry->isPast()) {
                        $newExpiry = Carbon::now()->addMonths($package->duration_months ?? 1);
                    } else {
                        $newExpiry = $currentExpiry->copy()->addMonths($package->duration_months ?? 1);
                    }
                    
                    $targetUser->update([
                        'package_id' => $packageId,
                        'license_expires_at' => $newExpiry,
                    ]);
                    
                    Log::info('Bank transfer payment approved', [
                        'transaction_id' => $transaction->id,
                        'user_id' => $userId,
                        'package_id' => $packageId,
                        'approved_by' => $user->id
                    ]);
                }
            }
        }
        
        // Transaction'ı tamamlandı olarak işaretle
        $transaction->update([
            'status' => 'completed',
        ]);
        
        return redirect()->back()->with('success', 'Ödeme onaylandı ve paket aktif edildi.');
    }

    public function reject(Request $request, $id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $transaction = Transaction::findOrFail($id);
        
        if ($transaction->status !== 'pending') {
            return redirect()->back()->with('error', 'Bu ödeme zaten işlenmiş.');
        }
        
        $transaction->update([
            'status' => 'failed',
        ]);
        
        Log::info('Bank transfer payment rejected', [
            'transaction_id' => $transaction->id,
            'rejected_by' => $user->id
        ]);
        
        return redirect()->back()->with('success', 'Ödeme reddedildi.');
    }
}
