<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Reservation;
use App\Models\Company;
use App\Models\Package;
use App\Models\Setting;
use App\Models\SalonRating;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class SuperAdminController extends Controller
{
    public function dashboard()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $stats = [
            'total_users' => User::where('role', 'güzellik_salonu')->count(),
            'active_users' => User::where('role', 'güzellik_salonu')
                ->whereNotNull('license_expires_at')
                ->where('license_expires_at', '>', now())
                ->count(),
            'total_appointments' => Reservation::count(),
            'pending_payments' => \App\Models\Transaction::where('status', 'pending')
                ->where('payment_method', 'bank_transfer')
                ->where('type', 'income')
                ->count(),
        ];
        
        return view('superadmin.dashboard', compact('stats'));
    }

    public function users()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $users = User::where('role', 'güzellik_salonu')
            ->with('package')
            ->orderBy('created_at', 'desc')
            ->paginate(20);
        
        return view('superadmin.users.index', compact('users'));
    }

    public function createUser()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $companies = Company::where('is_active', true)->orderBy('name')->get();
        
        return view('superadmin.users.create', compact('companies'));
    }

    public function storeUser(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'phone' => 'required|string|max:20',
            'password' => 'required|string|min:8|confirmed',
        ]);

        $newUser = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
            'password' => Hash::make($validated['password']),
            'role' => 'güzellik_salonu',
            'license_expires_at' => now()->addMonth(),
        ]);

        return redirect()->route('admin.superadmin.users')
            ->with('success', 'Kullanıcı başarıyla oluşturuldu.');
    }

    public function showUser($id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $targetUser = User::findOrFail($id);
        
        return view('superadmin.users.show', compact('targetUser'));
    }

    public function editUser($id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $targetUser = User::findOrFail($id);
        $companies = Company::where('is_active', true)->orderBy('name')->get();
        $packages = Package::where('is_active', true)->orderBy('sort_order')->orderBy('name')->get();
        
        return view('superadmin.users.edit', [
            'user' => $targetUser,
            'companies' => $companies,
            'packages' => $packages
        ]);
    }

    public function updateUser(Request $request, $id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $targetUser = User::findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $id,
            'phone' => 'required|string|max:20',
            'password' => 'nullable|string|min:8|confirmed',
            'role' => 'nullable|in:super_admin,admin,güzellik_salonu',
            'package_id' => 'nullable|exists:packages,id',
        ]);

        $updateData = [
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
        ];

        if ($request->filled('role')) {
            $updateData['role'] = $validated['role'];
        }

        if ($request->filled('password')) {
            $updateData['password'] = Hash::make($validated['password']);
        }

        if ($request->filled('package_id')) {
            $package = Package::findOrFail($validated['package_id']);
            $durationMonths = $package->duration_months ?? 1;

            $currentExpiry = $targetUser->license_expires_at ? Carbon::parse($targetUser->license_expires_at) : Carbon::now();

            if ($currentExpiry->isPast()) {
                $newExpiry = Carbon::now()->addMonths($durationMonths);
            } else {
                $newExpiry = $currentExpiry->copy()->addMonths($durationMonths);
            }
            
            $updateData['license_expires_at'] = $newExpiry;
            $updateData['package_id'] = $package->id;
        }

        $targetUser->update($updateData);

        return redirect()->route('admin.superadmin.users')
            ->with('success', 'Kullanıcı başarıyla güncellendi.');
    }

    public function destroyUser($id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $targetUser = User::findOrFail($id);
        
        if ($targetUser->role === 'super_admin') {
            return redirect()->route('admin.superadmin.users')
                ->with('error', 'Süper admin kullanıcı silinemez.');
        }
        if ($targetUser->id === $user->id) {
            return redirect()->route('admin.superadmin.users')
                ->with('error', 'Kendinizi silemezsiniz.');
        }
        
        try {
            $targetUser->delete();
        } catch (\Exception $e) {
            \Log::error('User delete error: ' . $e->getMessage());
            return redirect()->route('admin.superadmin.users')
                ->with('error', 'Kullanıcı silinirken hata: ' . $e->getMessage());
        }

        return redirect()->route('admin.superadmin.users')
            ->with('success', 'Kullanıcı başarıyla silindi.');
    }

    public function companies()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $companies = User::where('role', 'güzellik_salonu')
            ->orderBy('created_at', 'desc')
            ->paginate(20);
        
        return view('superadmin.companies.index', compact('companies'));
    }

    public function createCompany()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        return view('superadmin.companies.create');
    }

    public function storeCompany(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'phone' => 'required|string|max:20',
            'password' => 'required|string|min:8|confirmed',
        ]);

        $newCompany = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
            'password' => Hash::make($validated['password']),
            'role' => 'güzellik_salonu',
            'license_expires_at' => now()->addMonth(),
        ]);

        return redirect()->route('admin.superadmin.companies')
            ->with('success', 'Firma başarıyla oluşturuldu.');
    }

    public function showCompany($id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $company = User::findOrFail($id);
        
        return view('superadmin.companies.show', compact('company'));
    }

    public function editCompany($id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $company = User::findOrFail($id);
        
        return view('superadmin.companies.edit', compact('company'));
    }

    public function updateCompany(Request $request, $id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $company = User::findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $id,
            'phone' => 'required|string|max:20',
            'password' => 'nullable|string|min:8|confirmed',
        ]);

        $updateData = [
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
        ];

        if ($request->filled('password')) {
            $updateData['password'] = Hash::make($validated['password']);
        }

        $company->update($updateData);

        return redirect()->route('admin.superadmin.companies')
            ->with('success', 'Firma başarıyla güncellendi.');
    }

    public function destroyCompany($id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $company = User::findOrFail($id);
        $company->delete();

        return redirect()->route('admin.superadmin.companies')
            ->with('success', 'Firma başarıyla silindi.');
    }

    public function pricingSettings()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        return view('superadmin.pricing-settings');
    }

    public function updatePricingSettings(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        return redirect()->route('admin.superadmin.pricing-settings')
            ->with('success', 'Ayarlar başarıyla güncellendi.');
    }

    public function appDownloads()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $settings = Setting::where('group', 'apps')
            ->whereIn('key', ['windows_download_url', 'android_download_url', 'ios_download_url'])
            ->pluck('value', 'key')
            ->toArray();
        
        $downloadUrls = [
            'windows_download_url' => $settings['windows_download_url'] ?? '',
            'android_download_url' => $settings['android_download_url'] ?? '',
            'ios_download_url' => $settings['ios_download_url'] ?? '',
        ];
        
        return view('superadmin.app-downloads', compact('downloadUrls'));
    }

    public function updateAppDownloads(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $request->validate([
            'windows_download_url' => 'nullable|url',
            'android_download_url' => 'nullable|url',
            'ios_download_url' => 'nullable|url',
        ]);
        
        Setting::updateOrCreate(
            ['key' => 'windows_download_url', 'group' => 'apps'],
            ['value' => $request->windows_download_url ?? '']
        );
        
        Setting::updateOrCreate(
            ['key' => 'android_download_url', 'group' => 'apps'],
            ['value' => $request->android_download_url ?? '']
        );
        
        Setting::updateOrCreate(
            ['key' => 'ios_download_url', 'group' => 'apps'],
            ['value' => $request->ios_download_url ?? '']
        );
        
        return redirect()->route('admin.superadmin.app-downloads')
            ->with('success', 'İndirme linkleri başarıyla güncellendi.');
    }

    public function testSms()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        return view('superadmin.test-sms');
    }

    public function sendTestSms(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        return back()->with('success', 'Test SMS gönderildi.');
    }

    public function sendTestOtp(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        return back()->with('success', 'Test OTP gönderildi.');
    }

    public function testWhatsApp()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $sessions = DB::table('whatsapp_sessions')
            ->where('is_active', true)
            ->orderBy('created_at', 'desc')
            ->get();
        
        $activeRegId = session('reg_id');
        
        return view('superadmin.test-whatsapp', compact('sessions', 'activeRegId'));
    }

    public function loginWithPhone(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        return back()->with('success', 'Telefon ile giriş başlatıldı.');
    }

    public function loginWithQr(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        return back()->with('success', 'QR kod ile giriş başlatıldı.');
    }

    public function checkLogin(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        return response()->json(['status' => 'pending']);
    }

    public function sendTestWhatsApp(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        return back()->with('success', 'Test WhatsApp mesajı gönderildi.');
    }

    public function getDevices()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $sessions = DB::table('whatsapp_sessions')
            ->where('is_active', true)
            ->orderBy('created_at', 'desc')
            ->get();
        
        return response()->json(['devices' => $sessions]);
    }

    public function salonRatings(Request $request)
    {
        $user = Auth::user();

        if (!$user->isSuperAdmin()) {
            abort(403);
        }

        $search = trim((string) $request->query('q', ''));

        $ratings = SalonRating::query()
            ->with('salon')
            ->when($search !== '', function ($query) use ($search) {
                $query->whereHas('salon', function ($salonQuery) use ($search) {
                    $salonQuery->where('salon_name', 'like', '%' . $search . '%')
                        ->orWhere('name', 'like', '%' . $search . '%');
                });
            })
            ->orderByDesc('created_at')
            ->paginate(25)
            ->withQueryString();

        return view('superadmin.salon-ratings.index', compact('ratings', 'search'));
    }

    public function updateSalonRating(Request $request, $id)
    {
        $user = Auth::user();

        if (!$user->isSuperAdmin()) {
            abort(403);
        }

        $validated = $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'is_used' => 'nullable|boolean',
            'comment' => 'nullable|string|max:1000',
        ], [
            'rating.required' => 'Yıldız puanı zorunludur.',
            'rating.min' => 'Yıldız puanı en az 1 olmalıdır.',
            'rating.max' => 'Yıldız puanı en fazla 5 olmalıdır.',
        ]);

        $rating = SalonRating::findOrFail($id);
        $rating->rating = (int) $validated['rating'];
        $rating->is_used = (bool) $request->boolean('is_used');
        $rating->comment = $validated['comment'] ?? null;
        $rating->save();

        return back()->with('success', 'Salon yıldız değerlendirmesi güncellendi.');
    }
}

