<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Testimonial;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class TestimonialController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }
        
        $testimonials = Testimonial::orderBy('sort_order', 'asc')
            ->orderBy('created_at', 'desc')
            ->get();
        
        return view('superadmin.testimonials.index', compact('testimonials'));
    }

    public function create()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }
        
        return view('superadmin.testimonials.create');
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403, 'Bu işlemi yapma yetkiniz yok.');
        }
        
        $validated = $request->validate([
            'customer_name' => 'required|string|max:255',
            'customer_title' => 'nullable|string|max:255',
            'company_name' => 'nullable|string|max:255',
            'comment' => 'required|string',
            'rating' => 'required|integer|min:1|max:5',
            'sort_order' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);
        
        $validated['is_active'] = $request->has('is_active') ? true : false;
        $validated['sort_order'] = $validated['sort_order'] ?? 0;
        
        $testimonial = Testimonial::create($validated);
        
        Log::info('Testimonial created', [
            'user_id' => $user->id,
            'testimonial_id' => $testimonial->id,
            'customer_name' => $testimonial->customer_name
        ]);
        
        return redirect()->route('admin.superadmin.testimonials.index')
            ->with('success', 'Müşteri yorumu başarıyla eklendi.');
    }

    public function edit($id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403, 'Bu sayfaya erişim yetkiniz yok.');
        }
        
        $testimonial = Testimonial::findOrFail($id);
        
        return view('superadmin.testimonials.edit', compact('testimonial'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403, 'Bu işlemi yapma yetkiniz yok.');
        }
        
        $testimonial = Testimonial::findOrFail($id);
        
        $validated = $request->validate([
            'customer_name' => 'required|string|max:255',
            'customer_title' => 'nullable|string|max:255',
            'company_name' => 'nullable|string|max:255',
            'comment' => 'required|string',
            'rating' => 'required|integer|min:1|max:5',
            'sort_order' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);
        
        $validated['is_active'] = $request->has('is_active') ? true : false;
        $validated['sort_order'] = $validated['sort_order'] ?? 0;
        
        $testimonial->update($validated);
        
        Log::info('Testimonial updated', [
            'user_id' => $user->id,
            'testimonial_id' => $testimonial->id,
            'customer_name' => $testimonial->customer_name
        ]);
        
        return redirect()->route('admin.superadmin.testimonials.index')
            ->with('success', 'Müşteri yorumu başarıyla güncellendi.');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403, 'Bu işlemi yapma yetkiniz yok.');
        }
        
        $testimonial = Testimonial::findOrFail($id);
        $customerName = $testimonial->customer_name;
        
        $testimonial->delete();
        
        Log::info('Testimonial deleted', [
            'user_id' => $user->id,
            'testimonial_id' => $id,
            'customer_name' => $customerName
        ]);
        
        return redirect()->route('admin.superadmin.testimonials.index')
            ->with('success', 'Müşteri yorumu başarıyla silindi.');
    }
}

