<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class UserManagementController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $users = User::where('role', 'güzellik_salonu')
            ->with('package')
            ->orderBy('created_at', 'desc')
            ->paginate(20);
        
        return view('superadmin.users.index', compact('users'));
    }

    public function updateSmsBalance(Request $request, $id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $targetUser = User::findOrFail($id);
        
        $validated = $request->validate([
            'sms_balance' => 'required|integer|min:0',
        ]);
        
        $targetUser->update([
            'sms_balance' => $validated['sms_balance'],
        ]);
        
        return redirect()->back()
            ->with('success', 'SMS bakiyesi başarıyla güncellendi.');
    }

    public function updateLicense(Request $request, $id)
    {
        $user = Auth::user();
        
        if (!$user->isSuperAdmin()) {
            abort(403);
        }
        
        $targetUser = User::findOrFail($id);
        
        $validated = $request->validate([
            'license_days' => 'nullable|integer|min:0',
            'license_unlimited' => 'nullable|boolean',
        ]);
        
        $isUnlimited = !empty($validated['license_unlimited']);
        
        if ($isUnlimited) {
            $targetUser->update(['license_expires_at' => null]);
            $message = 'Lisans süresiz olarak ayarlandı.';
        } else {
            $licenseDays = (int)($validated['license_days'] ?? 0);
            if ($licenseDays <= 0) {
                $targetUser->update(['license_expires_at' => Carbon::now()->subSecond()]);
                $message = 'Lisans süresi sonlandırıldı.';
            } else {
                $newExpiry = Carbon::now()->startOfDay()->addDays($licenseDays)->endOfDay();
                $targetUser->update(['license_expires_at' => $newExpiry]);
                $message = 'Lisans süresi başarıyla güncellendi.';
            }
        }
        
        return redirect()->back()
            ->with('success', $message);
    }
}
