<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class CheckPackageExpiration
{
    
    public function handle(Request $request, Closure $next): Response
    {
        if (!Auth::check()) {
            return $next($request);
        }

        $user = Auth::user();

        if ($user->isSuperAdmin()) {
            return $next($request);
        }

        // Personel hesapları için lisans kontrolü yapma
        if ($user->role === 'personel') {
            return $next($request);
        }

        // Salon profil sayfası lisans/ödeme akışından bağımsız erişilebilir olmalı.
        // Aksi halde canlı ortamda bazı kullanıcılar profil ve paket yönlendirmesi arasında döngüye girebiliyor.
        if ($request->routeIs('admin.beauty-salon.profile*')) {
            return $next($request);
        }

        if ($user->role === 'güzellik_salonu') {
            // Geçerli lisans kontrolü (trial + lisans süresi)
            $hasValidLicense = $user->hasValidLicense();

            // Lisans geçersizse (trial bitmiş veya lisans süresi dolmuş)
            if (!$hasValidLicense) {
                $hasPendingBankTransfer = \App\Models\Transaction::where('user_id', $user->id)
                    ->where('type', 'income')
                    ->where('status', 'pending')
                    ->where('payment_method', 'bank_transfer')
                    ->where('payment_reference', 'like', 'PKG_%')
                    ->exists();

                // Pending ödeme varsa sadece kişisel faturalar sayfası kullanılabilir
                $allowedRoutes = $hasPendingBankTransfer
                    ? ['admin.settings.personal-invoices', 'logout']
                    : [
                        'admin.packages.select',
                        'admin.packages.purchase',
                        'admin.packages.coupon.apply',
                        'admin.settings.personal-invoices',
                        'admin.settings.invoice-info',
                        'payment.package.*',
                        'payment.*',
                        'logout',
                    ];

                $isAllowedRoute = false;
                foreach ($allowedRoutes as $route) {
                    if ($request->routeIs($route)) {
                        $isAllowedRoute = true;
                        break;
                    }
                }

                if (!$isAllowedRoute) {
                    // İşlemleri engelle; ödeme bekliyorsa bekleme ekranı, değilse paket seçimi
                    session()->put('show_package_selection', !$hasPendingBankTransfer);
                    session()->put('show_payment_pending', $hasPendingBankTransfer);
                    
                    // GET istekleri için uygun sayfaya yönlendir
                    if ($request->isMethod('GET')) {
                        $message = $hasPendingBankTransfer
                            ? 'Havale/EFT ödeme talebiniz alındı. Ödemeniz onaylandıktan sonra paketiniz aktif olacaktır.'
                            : 'Paket satın almanız gerekmektedir. Devam edebilmek için lütfen bir paket seçin.';

                        $redirectRoute = $hasPendingBankTransfer
                            ? 'admin.settings.personal-invoices'
                            : 'admin.packages.select';

                        // Güvenlik: herhangi bir nedenle hedef route ile aynı route'a düşerse
                        // sonsuz redirect döngüsünü engelle.
                        if ($request->routeIs($redirectRoute)) {
                            return $next($request);
                        }

                        return redirect()->route($redirectRoute)
                            ->with('error', $message);
                    }
                    
                    // POST/PUT/DELETE istekleri için hata döndür
                    if (in_array($request->method(), ['POST', 'PUT', 'PATCH', 'DELETE'])) {
                        $message = $hasPendingBankTransfer
                            ? 'Havale/EFT ödeme talebiniz onay bekliyor. Ödemeniz onaylandıktan sonra işlemlere devam edebilirsiniz.'
                            : 'Paket satın almanız gerekmektedir. Lütfen bir paket seçin.';

                        return response()->json([
                            'success' => false,
                            'message' => $message,
                            'redirect' => $hasPendingBankTransfer
                                ? route('admin.settings.personal-invoices')
                                : route('admin.packages.select')
                        ], 403);
                    }
                }

                // Kullanıcı izinli route'tayken de bekleme modalı için session bayrağı güncel kalsın
                session()->put('show_payment_pending', $hasPendingBankTransfer);
                if ($hasPendingBankTransfer) {
                    session()->put('show_package_selection', false);
                }
            }
        }

        return $next($request);
    }
}
