<?php

namespace App\Http\Middleware;

use App\Models\Salon;
use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class RequireSalon
{
    /**
     * güzellik_salonu kullanıcıları salon oluşturmadan diğer sayfalara geçemez.
     * Dashboard hariç tüm admin sayfalarına erişim engellenir.
     */
    public function handle(Request $request, Closure $next): Response
    {
        if (!auth()->check()) {
            return $next($request);
        }

        $user = auth()->user();

        // Sadece güzellik_salonu rolü için kontrol
        if ($user->role !== 'güzellik_salonu') {
            return $next($request);
        }

        // Süper admin atla
        if (method_exists($user, 'isSuperAdmin') && $user->isSuperAdmin()) {
            return $next($request);
        }

        // Zaten dashboard'daysa veya register/update-info (form submit) ise geç
        if ($request->routeIs('admin.dashboard') || $request->routeIs('register.update-info')) {
            return $next($request);
        }

        // Paket seçimi sayfasına izin ver (lisans süresi dolmuş olabilir)
        if ($request->routeIs('admin.packages.*')) {
            return $next($request);
        }

        // Hesap, üyelik, fatura bilgileri, kişisel faturalar - paket/ödeme için gerekli
        if ($request->routeIs('admin.account.*') || $request->routeIs('admin.settings.membership') || $request->routeIs('admin.settings.invoice-info*') || $request->routeIs('admin.settings.personal-invoices')) {
            return $next($request);
        }

        // Salon profili sayfaları her zaman erişilebilir olmalı.
        // Aksi halde salonu olmayan kullanıcı profil sayfasına giremeyip dashboard'a döner
        // ve bazı kurulum akışlarında yönlendirme döngüsü oluşabilir.
        if ($request->routeIs('admin.beauty-salon.profile*')) {
            return $next($request);
        }

        // Salon var mı kontrol et
        $hasSalon = false;
        try {
            if (!empty($user->salon_name) || !empty($user->salon_slug)) {
                $hasSalon = true;
            } elseif (class_exists(Salon::class) && \Schema::hasTable('salons')) {
                $hasSalon = Salon::where('created_by', $user->id)->exists();
            }
        } catch (\Exception $e) {
            \Log::warning('RequireSalon: Salon kontrolü hatası - ' . $e->getMessage());
        }

        if (!$hasSalon) {
            // Güvenlik: hedef route mevcut route ise döngüye düşme
            if ($request->routeIs('admin.dashboard')) {
                return $next($request);
            }

            return redirect()->route('admin.dashboard')
                ->with('warning', 'Devam etmek için önce salon bilgilerinizi tamamlamanız gerekmektedir.');
        }

        return $next($request);
    }
}
