<?php

namespace App\Jobs;

use App\Models\Reservation;
use App\Models\NotificationDevice;
use App\Services\ApnsService;
use App\Services\NotificationService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class SendAppointmentReminderJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        private int $reservationId,
        private int $hoursBefore
    ) {
        $this->onQueue('notifications');
    }

    public function handle(ApnsService $apnsService, NotificationService $notificationService): void
    {
        $reservation = Reservation::with(['customer', 'service', 'personel', 'room'])
            ->find($this->reservationId);

        if (!$reservation) {
            Log::warning("Reservation not found for reminder: {$this->reservationId}");
            return;
        }

        $user = $reservation->beauty_salon_id 
            ? \App\Models\User::where('beauty_salon_id', $reservation->beauty_salon_id)->first()
            : $reservation->personel;

        if (!$user) {
            Log::warning("User not found for reservation reminder: {$this->reservationId}");
            return;
        }

        $preferences = $notificationService->getNotificationPreferences($user);
        if (!($preferences['appointment_reminders'] ?? true)) {
            return; 
        }

        $devices = NotificationDevice::where('user_id', $user->id)
            ->where('is_active', true)
            ->where('platform', 'ios')
            ->get();

        if ($devices->isEmpty()) {
            Log::info("No active devices for user: {$user->id}");
            return;
        }

        $customerName = $reservation->customer->name ?? 'Müşteri';
        $serviceName = $reservation->service->name ?? 'Hizmet';
        $time = $reservation->start_time;

        $title = "Randevu Hatırlatıcı";
        $body = "{$customerName} ile randevunuz {$this->hoursBefore} saat sonra ({$time})";

        $data = [
            'type' => 'appointment_reminder',
            'appointment_id' => $reservation->id,
            'hours_before' => $this->hoursBefore,
        ];

        $successCount = 0;
        foreach ($devices as $device) {
            if ($apnsService->sendNotification($device, $title, $body, $data)) {
                $successCount++;
            }
        }

        Log::info("Appointment reminder sent", [
            'reservation_id' => $this->reservationId,
            'hours_before' => $this->hoursBefore,
            'devices' => $devices->count(),
            'success' => $successCount,
        ]);
    }
}

