<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Schema;

class Account extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'type',
        'balance',
        'is_active',
        'user_id',
        'beauty_salon_id',
    ];

    protected $casts = [
        'balance' => 'decimal:2',
        'is_active' => 'boolean',
    ];

    public function transactions()
    {
        return $this->hasMany(Transaction::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }
    
    /**
     * Kasa bakiyesini işlemlere göre yeniden hesapla
     */
    public function recalculateBalance()
    {
        $income = $this->transactions()->where('type', 'income')->sum('amount') ?? 0;
        $expense = $this->transactions()->where('type', 'expense')->sum('amount') ?? 0;
        $newBalance = $income - $expense;
        
        $this->update(['balance' => $newBalance]);
        
        return $newBalance;
    }

    public static function getCashAccount($userId)
    {
        $user = \App\Models\User::find($userId);
        if (!$user) {
            throw new \Exception('Kullanıcı bulunamadı');
        }
        
        // Salon ID'sini belirle:
        // - Eğer kullanıcı personel ise beauty_salon_id kullan
        // - Eğer kullanıcı güzellik_salonu rolündeyse kendi id'si salon id'sidir
        // - Diğer durumlarda beauty_salon_id varsa onu kullan, yoksa kendi id'si
        $salonId = null;
        
        if ($user->isPersonel() && $user->beauty_salon_id) {
            $salonId = $user->beauty_salon_id;
        } elseif ($user->isGuzellikSalonu()) {
            $salonId = $user->id; // Güzellik salonu kullanıcısının kendi id'si salon id'sidir
        } elseif ($user->beauty_salon_id) {
            $salonId = $user->beauty_salon_id;
        } else {
            // Eğer hiçbir salon id'si yoksa, kullanıcının kendi id'sini salon id'si olarak kullan
            // (eski sistem uyumluluğu için)
            $salonId = $user->id;
        }
        
        if (!$salonId) {
            throw new \Exception('Salon ID belirlenemedi. Kullanıcı ID: ' . $userId);
        }
        
        // Eğer beauty_salon_id kolonu yoksa eski sistemi kullan
        if (!\Schema::hasColumn('accounts', 'beauty_salon_id')) {
            // Eski sistem: user_id ile
            if (!\Schema::hasColumn('accounts', 'user_id')) {
                return static::where('type', 'cash')->first() ?? static::create([
                    'name' => 'Kasa',
                    'type' => 'cash',
                    'balance' => 0,
                    'is_active' => true,
                ]);
            }
            
            return static::firstOrCreate(
                [
                    'user_id' => $userId,
                    'type' => 'cash',
                ],
                [
                    'name' => 'Kasa',
                    'balance' => 0,
                    'is_active' => true,
                ]
            );
        }
        
        // Yeni sistem: beauty_salon_id ile (salon bazlı)
        // Aynı salon için tek bir kasa hesabı olmalı
        $createData = [
            'name' => 'Kasa',
            'balance' => 0,
            'is_active' => true,
        ];
        
        // user_id kolonu varsa ekle
        if (\Schema::hasColumn('accounts', 'user_id')) {
            $createData['user_id'] = $userId;
        }
        
        $cashAccount = static::firstOrCreate(
            [
                'beauty_salon_id' => $salonId,
                'type' => 'cash',
            ],
            $createData
        );
        
        // Debug: Kasa hesabının doğru oluşturulduğundan emin ol
        if (!$cashAccount) {
            \Log::error('Kasa hesabı oluşturulamadı', [
                'user_id' => $userId,
                'salon_id' => $salonId,
                'user_role' => $user->role
            ]);
            throw new \Exception('Kasa hesabı oluşturulamadı. Salon ID: ' . $salonId);
        }
        
        return $cashAccount;
    }
}

