<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Contract extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'content',
        'is_active',
        'is_default',
        'is_global',
        'is_required',
        'public_token',
        'company_id',
        'user_id',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'is_default' => 'boolean',
        'is_global' => 'boolean',
        'is_required' => 'boolean',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($contract) {
            if (empty($contract->public_token)) {
                $contract->public_token = Str::random(32);
            }
        });
    }

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function getPublicUrlAttribute()
    {
        return url('/sozlesme/' . $this->public_token);
    }

    public static function getDefaultForCompany($companyId)
    {
        return static::where('company_id', $companyId)
            ->where('is_default', true)
            ->where('is_active', true)
            ->first();
    }

    public static function getGlobalContract()
    {
        return static::where(function($query) {
                $query->where('is_global', true)
                      ->orWhere('is_required', true);
            })
            ->where('is_active', true)
            ->orderBy('is_global', 'desc')
            ->orderBy('is_required', 'desc')
            ->first();
    }

    public function setAsDefault()
    {
        
        static::where('user_id', $this->user_id)
            ->where('id', '!=', $this->id)
            ->update(['is_default' => false]);

        $this->update(['is_default' => true]);
    }
}

