<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Coupon extends Model
{
    protected $fillable = [
        'code',
        'name',
        'type',
        'subscription_days',
        'sms_credits',
        'max_uses',
        'used_count',
        'valid_from',
        'valid_until',
        'is_active',
    ];

    protected function casts(): array
    {
        return [
            'valid_from' => 'datetime',
            'valid_until' => 'datetime',
            'is_active' => 'boolean',
        ];
    }

    public const TYPE_SUBSCRIPTION = 'subscription';
    public const TYPE_SMS = 'sms';
    public const TYPE_BOTH = 'both';

    public function redemptions(): HasMany
    {
        return $this->hasMany(CouponRedemption::class);
    }

    public function isValid(): bool
    {
        if (!$this->is_active) {
            return false;
        }
        if ($this->valid_from && now()->lt($this->valid_from)) {
            return false;
        }
        if ($this->valid_until && now()->gt($this->valid_until)) {
            return false;
        }
        if ($this->max_uses !== null && $this->used_count >= $this->max_uses) {
            return false;
        }
        return true;
    }

    public function hasSubscriptionBenefit(): bool
    {
        return in_array($this->type, [self::TYPE_SUBSCRIPTION, self::TYPE_BOTH], true)
            && $this->subscription_days > 0;
    }

    public function hasSmsBenefit(): bool
    {
        return in_array($this->type, [self::TYPE_SMS, self::TYPE_BOTH], true)
            && $this->sms_credits > 0;
    }
}
