<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class SalonRating extends Model
{
    protected $fillable = [
        'salon_id',
        'customer_id',
        'reservation_id',
        'rating',
        'comment',
        'token',
        'referral_code',
        'expires_at',
        'is_used',
        'customer_name',
        'customer_phone',
    ];

    public static function generateReferralCode(): string
    {
        do {
            $code = str_pad(rand(100000, 999999), 6, '0', STR_PAD_LEFT);
        } while (self::where('referral_code', $code)->exists());
        
        return $code;
    }

    protected $casts = [
        'rating' => 'integer',
        'is_used' => 'boolean',
        'expires_at' => 'datetime',
    ];

    public static function generateToken(): string
    {
        do {
            $token = Str::random(12);
        } while (self::where('token', $token)->exists());
        
        return $token;
    }

    public function salon()
    {
        return $this->belongsTo(User::class, 'salon_id');
    }

    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    public function reservation()
    {
        return $this->belongsTo(Reservation::class);
    }

    public function isValid(): bool
    {
        if ($this->is_used) {
            return false;
        }
        
        if ($this->expires_at && $this->expires_at->isPast()) {
            return false;
        }
        
        return true;
    }

    public static function getAverageRating($salonId): float
    {
        return self::where('salon_id', $salonId)
            ->where('is_used', true)
            ->where('rating', '>', 0)
            ->avg('rating') ?? 0;
    }

    public static function getRatingCount($salonId): int
    {
        return self::where('salon_id', $salonId)
            ->where('is_used', true)
            ->where('rating', '>', 0)
            ->count();
    }
}
