<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    use HasFactory, Notifiable, HasApiTokens;

    protected $fillable = [
        'name',
        'email',
        'password',
        'role',
        'phone',
        'is_active',
        'company_id',
        'beauty_salon_id',
        'is_personel_admin',
        'profile_image',
        'salon_name',
        'salon_short_name',
        'salon_slug',
        'short_url',
        'salon_phone',
        'salon_address',
        'salon_city',
        'salon_district',
        'salon_profile_image',
        'salon_banner_image',
        'salon_theme',
        'salon_color',
        'salon_description',
        'salon_working_hours',
        'salon_gallery',
        'personel_tags',
        'sms_balance',
        'license_key',
        'license_expires_at',
        'package_id',
        'trial_ends_at',
        'free_access_ends_at',
        'kvkk_approved_at',
        'privacy_approved_at',
        'terms_approved_at',
        'pricing_notice_approved_at',
        'consent_ip_address',
        'consent_user_agent',
        'invoice_type',
        'invoice_name',
        'invoice_tax_number',
        'invoice_tax_office',
        'invoice_address',
        'invoice_city',
        'invoice_district',
        'invoice_postal_code',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'is_active' => 'boolean',
            'is_personel_admin' => 'boolean',
            'personel_tags' => 'array',
            'salon_gallery' => 'array',
            'salon_working_hours' => 'array',
            'license_expires_at' => 'datetime',
            'trial_ends_at' => 'datetime',
            'free_access_ends_at' => 'datetime',
            'kvkk_approved_at' => 'datetime',
            'privacy_approved_at' => 'datetime',
            'terms_approved_at' => 'datetime',
            'pricing_notice_approved_at' => 'datetime',
        ];
    }

    public function isSuperAdmin(): bool
    {
        return $this->role === 'super_admin';
    }

    public function isBeautySalon(): bool
    {
        return $this->role === 'güzellik_salonu';
    }

    public function isGuzellikSalonu(): bool
    {
        return $this->role === 'güzellik_salonu';
    }

    public function isPersonel(): bool
    {
        return $this->role === 'personel';
    }

    public function isPersonelAdmin(): bool
    {
        return $this->isPersonel() && $this->is_personel_admin === true;
    }

    public function getInitialsAttribute(): string
    {
        $name = $this->salon_name ?? $this->name;
        $words = explode(' ', $name);
        $initials = '';
        foreach ($words as $word) {
            if (!empty($word)) {
                $initials .= mb_substr($word, 0, 1, 'UTF-8');
            }
        }
        return mb_strtoupper(mb_substr($initials, 0, 2, 'UTF-8'), 'UTF-8');
    }

    public function getAvatarColorAttribute(): string
    {
        $colors = [
            '#ec4899', '#3b82f6', '#8b5cf6', '#f59e0b', '#10b981',
            '#ef4444', '#06b6d4', '#f97316', '#84cc16', '#6366f1'
        ];
        $index = crc32($this->email ?? $this->id) % count($colors);
        return $colors[$index];
    }

    public function beautySalon()
    {
        return $this->belongsTo(User::class, 'beauty_salon_id');
    }

    public function personels()
    {
        return $this->hasMany(User::class, 'beauty_salon_id');
    }

    public function services()
    {
        return $this->belongsToMany(\App\Models\Service::class, 'personel_service', 'personel_id', 'service_id');
    }

    public function hasSmsBalance(int $required = 1): bool
    {
        
        if ($this->isSuperAdmin()) {
            return true;
        }

        return ($this->sms_balance ?? 0) >= $required;
    }

    public function deductSmsBalance(int $amount = 1): bool
    {
        if ($this->isSuperAdmin()) {
            return true; 
        }

        if (!$this->hasSmsBalance($amount)) {
            return false;
        }

        $this->sms_balance = max(0, ($this->sms_balance ?? 0) - $amount);
        return $this->save();
    }

    public function hasValidLicense(): bool
    {
        if ($this->isSuperAdmin()) {
            return true;
        }

        // Personel hesapları için lisans kontrolü yapma
        if ($this->role === 'personel') {
            return true;
        }

        // Kupon veya kampanya ile verilen ücretsiz süre varsa paket kontrolü yapılmaz.
        if ($this->free_access_ends_at && now()->lt($this->free_access_ends_at)) {
            return true;
        }

        // Trial kontrolü
        $trialExpired = $this->trial_ends_at && now()->gte($this->trial_ends_at);
        $noPackage = !$this->package_id;
        
        // Trial bitmiş ve paket yoksa
        if ($trialExpired && $noPackage) {
            return false;
        }

        // Lisans süresi kontrolü
        if ($this->license_expires_at === null) {
            // Lisans süresi null ise ve paket varsa, paket süresine göre kontrol et
            if ($noPackage) {
                return false;
            }
            return true;
        }

        $now = now();
        $expiresAt = \Carbon\Carbon::parse($this->license_expires_at);

        return $now->lt($expiresAt);
    }

    public function hasActivePackage(): bool
    {
        if ($this->isSuperAdmin()) {
            return true;
        }

        if ($this->free_access_ends_at && now()->lt($this->free_access_ends_at)) {
            return true;
        }

        // Trial kontrolü
        $trialActive = !$this->trial_ends_at || now()->lt($this->trial_ends_at);
        
        // Trial aktifse veya paket varsa
        if ($trialActive || $this->package_id) {
            return true;
        }

        return false;
    }

    public function package()
    {
        return $this->belongsTo(Package::class);
    }

}

