<?php

namespace App\Services;

use App\Models\NotificationDevice;
use App\Models\NotificationLog;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class ApnsService
{
    private string $apnsUrl;
    private string $apnsKeyId;
    private string $apnsTeamId;
    private string $apnsBundleId;
    private string $apnsKeyPath;

    public function __construct()
    {
        $this->apnsUrl = config('services.apns.url', 'https:
        $this->apnsKeyId = config('services.apns.key_id');
        $this->apnsTeamId = config('services.apns.team_id');
        $this->apnsBundleId = config('services.apns.bundle_id');
        $this->apnsKeyPath = config('services.apns.key_path');
    }

    public function isConfigured(): bool
    {
        return !empty($this->apnsKeyId) 
            && !empty($this->apnsTeamId) 
            && !empty($this->apnsBundleId)
            && file_exists($this->apnsKeyPath);
    }

    public function sendNotification(
        NotificationDevice $device,
        string $title,
        string $body,
        array $data = [],
        string $sound = 'default',
        int $badge = 1
    ): bool {
        
        if (!$this->isConfigured()) {
            Log::warning('APNs is not configured. Skipping notification.', [
                'device_id' => $device->id,
            ]);
            return false;
        }

        try {
            $payload = $this->buildPayload($title, $body, $data, $sound, $badge);
            $token = $this->generateJwtToken();

            $response = Http::withHeaders([
                'Authorization' => "Bearer {$token}",
                'apns-push-type' => 'alert',
                'apns-priority' => '10',
                'apns-topic' => $this->apnsBundleId,
            ])->post("{$this->apnsUrl}/3/device/{$device->device_token}", $payload);

            $status = $response->successful();
            $statusCode = $response->status();

            NotificationLog::create([
                'user_id' => $device->user_id,
                'notification_device_id' => $device->id,
                'type' => $data['type'] ?? 'unknown',
                'title' => $title,
                'message' => $body,
                'data' => $data,
                'channel' => 'push',
                'status' => $status ? 'sent' : 'failed',
                'error_message' => $status ? null : $response->body(),
                'sent_at' => now(),
            ]);

            $device->update(['last_seen_at' => now()]);

            if (!$status) {
                Log::error('APNs notification failed', [
                    'device_id' => $device->id,
                    'status_code' => $statusCode,
                    'response' => $response->body(),
                ]);

                if ($statusCode === 410) {
                    $device->update(['is_active' => false]);
                }
            }

            return $status;
        } catch (\Exception $e) {
            Log::error('APNs service error', [
                'error' => $e->getMessage(),
                'device_id' => $device->id,
            ]);

            NotificationLog::create([
                'user_id' => $device->user_id,
                'notification_device_id' => $device->id,
                'type' => $data['type'] ?? 'unknown',
                'title' => $title,
                'message' => $body,
                'data' => $data,
                'channel' => 'push',
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);

            return false;
        }
    }

    private function buildPayload(string $title, string $body, array $data, string $sound, int $badge): array
    {
        return [
            'aps' => [
                'alert' => [
                    'title' => $title,
                    'body' => $body,
                ],
                'sound' => $sound,
                'badge' => $badge,
                'content-available' => 1, 
            ],
            ...$data, 
        ];
    }

    private function generateJwtToken(): string
    {
        if (!file_exists($this->apnsKeyPath)) {
            throw new \Exception("APNs key file not found: {$this->apnsKeyPath}");
        }

        $header = [
            'alg' => 'ES256',
            'kid' => $this->apnsKeyId,
        ];

        $claims = [
            'iss' => $this->apnsTeamId,
            'iat' => time(),
        ];

        $headerEncoded = $this->base64UrlEncode(json_encode($header));
        $claimsEncoded = $this->base64UrlEncode(json_encode($claims));

        $signature = '';
        openssl_sign(
            "{$headerEncoded}.{$claimsEncoded}",
            $signature,
            file_get_contents($this->apnsKeyPath),
            OPENSSL_ALGO_SHA256
        );

        $signatureEncoded = $this->base64UrlEncode($signature);

        return "{$headerEncoded}.{$claimsEncoded}.{$signatureEncoded}";
    }

    private function base64UrlEncode(string $data): string
    {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }
}

