<?php

namespace App\Services;

use App\Models\User;
use App\Models\Reservation;
use App\Models\Invoice;
use App\Models\NotificationDevice;
use App\Jobs\SendAppointmentReminderJob;
use Illuminate\Support\Facades\Log;

class NotificationService
{
    private ApnsService $apnsService;

    public function __construct(ApnsService $apnsService)
    {
        $this->apnsService = $apnsService;
    }

    public function sendAppointmentReminder(Reservation $reservation, int $hoursBefore): void
    {
        SendAppointmentReminderJob::dispatch($reservation->id, $hoursBefore);
    }

    public function sendAppointmentApproved(Reservation $reservation): void
    {
        $customer = $reservation->customer;
        if (!$customer || !$customer->phone) {
            return;
        }

        $user = User::where('phone', $customer->phone)->first();
        if (!$user) {
            return; 
        }

        $devices = NotificationDevice::where('user_id', $user->id)
            ->where('is_active', true)
            ->get();

        $title = "Randevu Onaylandı";
        $body = "Randevunuz onaylandı. Tarih: {$reservation->date->format('d.m.Y')} Saat: {$reservation->start_time}";

        $data = [
            'type' => 'appointment_approved',
            'appointment_id' => $reservation->id,
        ];

        if ($this->apnsService->isConfigured()) {
            foreach ($devices as $device) {
                $this->apnsService->sendNotification($device, $title, $body, $data);
            }
        }

        $user->notify(new \App\Notifications\AppointmentApproved($reservation));
    }

    public function sendAppointmentCancelled(Reservation $reservation): void
    {
        $customer = $reservation->customer;
        if (!$customer || !$customer->phone) {
            return;
        }

        $user = User::where('phone', $customer->phone)->first();
        if (!$user) {
            return;
        }

        $devices = NotificationDevice::where('user_id', $user->id)
            ->where('is_active', true)
            ->get();

        $title = "Randevu İptal Edildi";
        $body = "Randevunuz iptal edildi. Tarih: {$reservation->date->format('d.m.Y')} Saat: {$reservation->start_time}";

        $data = [
            'type' => 'appointment_cancelled',
            'appointment_id' => $reservation->id,
        ];

        if ($this->apnsService->isConfigured()) {
            foreach ($devices as $device) {
                $this->apnsService->sendNotification($device, $title, $body, $data);
            }
        }

        $user->notify(new \App\Notifications\AppointmentCancelled($reservation));
    }

    public function sendInvoiceCreated(Invoice $invoice): void
    {
        $customer = $invoice->customer;
        if (!$customer || !$customer->phone) {
            return;
        }

        $user = User::where('phone', $customer->phone)->first();
        if (!$user) {
            return;
        }

        $devices = NotificationDevice::where('user_id', $user->id)
            ->where('is_active', true)
            ->get();

        $title = "Yeni Fatura";
        $body = "Faturanız oluşturuldu. Tutar: {$invoice->total_amount} ₺";

        $data = [
            'type' => 'invoice_created',
            'invoice_id' => $invoice->id,
        ];

        if ($this->apnsService->isConfigured()) {
            foreach ($devices as $device) {
                $this->apnsService->sendNotification($device, $title, $body, $data);
            }
        }

        $user->notify(new \App\Notifications\InvoiceCreated($invoice));
    }

    public function getNotificationPreferences(User $user): array
    {
        $settings = \App\Models\Setting::where('user_id', $user->id)
            ->where('group', 'notifications')
            ->pluck('value', 'key')
            ->toArray();

        return [
            'appointment_reminders' => $settings['appointment_reminders'] ?? true,
            'appointment_approved' => $settings['appointment_approved'] ?? true,
            'appointment_cancelled' => $settings['appointment_cancelled'] ?? true,
            'invoice_created' => $settings['invoice_created'] ?? true,
            'payment_reminder' => $settings['payment_reminder'] ?? true,
            'system_announcements' => $settings['system_announcements'] ?? true,
        ];
    }
}

