<?php

namespace App\Services\Payment;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class IyzicoService
{
    protected $apiKey;
    protected $secretKey;
    protected $baseUrl;

    public function __construct()
    {
        $settings = \App\Models\Setting::where('group', 'payment')->pluck('value', 'key')->toArray();
        $this->apiKey = $settings['iyzico_api_key'] ?? config('payment.iyzico.api_key');
        $this->secretKey = $settings['iyzico_secret_key'] ?? config('payment.iyzico.secret_key');
        $testMode = $settings['iyzico_test_mode'] ?? config('payment.iyzico.test_mode', true);
        $this->baseUrl = ($testMode == '1' || $testMode === true) 
            ? 'https://sandbox-api.iyzipay.com'
            : 'https://api.iyzipay.com';
    }

    public function createPayment($params)
    {
        try {
            $paymentAmount = $params['payment_amount'] ?? $params['price'] ?? 0;
            $callbackUrl = $params['callback_url'] ?? $params['merchant_ok_url'] ?? route('payment.callback');
            
            $request = [
                'locale' => 'tr',
                'conversationId' => $params['merchant_oid'] ?? $params['conversation_id'] ?? uniqid('iyzico_'),
                'price' => number_format($paymentAmount, 2, '.', ''),
                'paidPrice' => number_format($paymentAmount, 2, '.', ''),
                'currency' => $params['currency'] ?? 'TRY',
                'basketId' => $params['basket_id'] ?? uniqid('basket_'),
                'paymentGroup' => 'PRODUCT',
                'callbackUrl' => $callbackUrl,
                'buyer' => [
                    'id' => $params['buyer_id'] ?? 'BY' . uniqid(),
                    'name' => $params['buyer_name'] ?? '',
                    'surname' => $params['buyer_surname'] ?? '',
                    'gsmNumber' => $params['buyer_phone'] ?? '',
                    'email' => $params['email'] ?? $params['buyer_email'] ?? '',
                    'identityNumber' => $params['buyer_identity'] ?? '11111111111',
                    'lastLoginDate' => date('Y-m-d H:i:s'),
                    'registrationDate' => date('Y-m-d H:i:s'),
                    'registrationAddress' => $params['buyer_address'] ?? '',
                    'ip' => $this->getUserIp(),
                    'city' => $params['buyer_city'] ?? 'Istanbul',
                    'country' => 'Turkey',
                    'zipCode' => $params['buyer_zip'] ?? '34000'
                ],
                'shippingAddress' => [
                    'contactName' => $params['buyer_name'] ?? '',
                    'city' => $params['buyer_city'] ?? 'Istanbul',
                    'country' => 'Turkey',
                    'address' => $params['buyer_address'] ?? '',
                    'zipCode' => $params['buyer_zip'] ?? '34000'
                ],
                'billingAddress' => [
                    'contactName' => $params['buyer_name'] ?? '',
                    'city' => $params['buyer_city'] ?? 'Istanbul',
                    'country' => 'Turkey',
                    'address' => $params['buyer_address'] ?? '',
                    'zipCode' => $params['buyer_zip'] ?? '34000'
                ],
                'basketItems' => [
                    [
                        'id' => $params['product_id'] ?? 'PRD' . uniqid(),
                        'name' => $params['product_name'] ?? 'Ürün',
                        'category1' => $params['product_category'] ?? 'Genel',
                        'itemType' => 'PHYSICAL',
                        'price' => number_format($paymentAmount, 2, '.', '')
                    ]
                ]
            ];

            $authorization = $this->generateAuthorization('POST', '/payment/checkoutform/initialize', $request);
            
            $response = Http::withHeaders([
                'Authorization' => $authorization,
                'Content-Type' => 'application/json'
            ])->post($this->baseUrl . '/payment/checkoutform/initialize', $request);

            $result = $response->json();

            if (isset($result['status']) && $result['status'] == 'success') {
                return [
                    'success' => true,
                    'checkout_form_content' => $result['checkoutFormContent'],
                    'payment_page_url' => $result['paymentPageUrl'] ?? null,
                    'conversation_id' => $result['conversationId'] ?? $request['conversationId']
                ];
            }

            Log::error('Iyzico payment creation failed', ['response' => $result, 'params' => $params]);
            return [
                'success' => false,
                'message' => $result['errorMessage'] ?? 'Ödeme başlatılamadı',
                'error_code' => $result['errorCode'] ?? null
            ];
        } catch (\Exception $e) {
            Log::error('Iyzico payment creation exception', ['error' => $e->getMessage(), 'params' => $params]);
            return [
                'success' => false,
                'message' => 'Ödeme başlatılamadı: ' . $e->getMessage()
            ];
        }
    }

    public function verifyCallback($post)
    {
        if (!isset($post['token'])) {
            return ['success' => false, 'message' => 'Token bulunamadı'];
        }

        $request = [
            'locale' => 'tr',
            'token' => $post['token']
        ];

        $authorization = $this->generateAuthorization('POST', '/payment/checkoutform/auth', $request);
        
        $response = Http::withHeaders([
            'Authorization' => $authorization,
            'Content-Type' => 'application/json'
        ])->post($this->baseUrl . '/payment/checkoutform/auth', $request);

        $result = $response->json();

        if (isset($result['status']) && $result['status'] == 'success' && isset($result['paymentStatus']) && $result['paymentStatus'] == 'SUCCESS') {
            return [
                'success' => true,
                'merchant_oid' => $result['conversationId'] ?? null,
                'conversation_id' => $result['conversationId'] ?? null,
                'payment_id' => $result['paymentId'] ?? null,
                'price' => $result['price'] ?? null,
                'paid_price' => $result['paidPrice'] ?? null,
                'total_amount' => $result['paidPrice'] ?? $result['price'] ?? null,
                'currency' => $result['currency'] ?? 'TRY'
            ];
        }

        return [
            'success' => false,
            'message' => $result['errorMessage'] ?? 'Ödeme doğrulama başarısız',
            'error_code' => $result['errorCode'] ?? null
        ];
    }

    protected function generateAuthorization($method, $path, $request)
    {
        $randomString = $this->generateRandomString();
        $requestString = json_encode($request, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        
        $hash = base64_encode(sha1($this->apiKey . $randomString . $this->secretKey . $requestString, true));
        
        return 'IYZWS ' . $this->apiKey . ':' . $hash . ':' . $randomString;
    }

    protected function generateRandomString($length = 8)
    {
        return substr(str_shuffle(str_repeat($x='0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ', ceil($length/strlen($x)) )),1,$length);
    }

    protected function getUserIp()
    {
        if (isset($_SERVER["HTTP_CLIENT_IP"])) {
            return $_SERVER["HTTP_CLIENT_IP"];
        } elseif (isset($_SERVER["HTTP_X_FORWARDED_FOR"])) {
            return $_SERVER["HTTP_X_FORWARDED_FOR"];
        } else {
            return $_SERVER["REMOTE_ADDR"] ?? '127.0.0.1';
        }
    }
}

