<?php

namespace App\Services\Payment;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class PayTRService
{
    protected $merchantId;
    protected $merchantKey;
    protected $merchantSalt;
    protected $testMode;

    public function __construct()
    {
        $settings = \App\Models\Setting::where('group', 'payment')->pluck('value', 'key')->toArray();
        $this->merchantId = $settings['paytr_merchant_id'] ?? config('payment.paytr.merchant_id');
        $this->merchantKey = $settings['paytr_merchant_key'] ?? config('payment.paytr.merchant_key');
        $this->merchantSalt = $settings['paytr_merchant_salt'] ?? config('payment.paytr.merchant_salt');
        $this->testMode = $settings['paytr_test_mode'] ?? config('payment.paytr.test_mode', '0');
    }

    public function createPayment($params)
    {
        try {
            $merchantOid = $params['merchant_oid'] ?? uniqid('paytr_');
            $email = $params['email'];
            $paymentAmount = number_format($params['payment_amount'], 2, '.', '');
            $userIp = $this->getUserIp();
            $userBasket = $params['user_basket'] ?? base64_encode(json_encode([[$params['product_name'] ?? 'Ürün', $paymentAmount, 1]]));
            $paymentType = $params['payment_type'] ?? 'card';
            $currency = $params['currency'] ?? 'TL';
            $non3d = $params['non_3d'] ?? '0';
            $installmentCount = $params['installment_count'] ?? '0';
            $merchantOkUrl = $params['merchant_ok_url'];
            $merchantFailUrl = $params['merchant_fail_url'];
            $clientLang = $params['client_lang'] ?? 'tr';

            $hashStr = $this->merchantId . $userIp . $merchantOid . $email . $paymentAmount . $paymentType . $installmentCount . $currency . $this->testMode . $non3d;
            $token = base64_encode(hash_hmac('sha256', $hashStr . $this->merchantSalt, $this->merchantKey, true));

            return [
                'success' => true,
                'merchant_id' => $this->merchantId,
                'user_ip' => $userIp,
                'merchant_oid' => $merchantOid,
                'email' => $email,
                'payment_type' => $paymentType,
                'payment_amount' => $paymentAmount,
                'currency' => $currency,
                'test_mode' => $this->testMode,
                'non_3d' => $non3d,
                'merchant_ok_url' => $merchantOkUrl,
                'merchant_fail_url' => $merchantFailUrl,
                'user_basket' => $userBasket,
                'client_lang' => $clientLang,
                'paytr_token' => $token,
                'installment_count' => $installmentCount,
                'post_url' => 'https://www.paytr.com/odeme/api/get-token',
            ];
        } catch (\Exception $e) {
            Log::error('PayTR payment creation failed', ['error' => $e->getMessage(), 'params' => $params]);
            return [
                'success' => false,
                'message' => 'Ödeme başlatılamadı: ' . $e->getMessage()
            ];
        }
    }

    /**
     * PayTR Direkt API - Kart bilgileri kendi sayfamızda girilir
     */
    public function getDirectPaymentFormData(array $params): array
    {
        try {
            $merchantOid = $params['merchant_oid'] ?? uniqid('paytr_');
            $email = $params['email'];
            $paymentAmount = number_format((float) $params['payment_amount'], 2, '.', '');
            $userIp = $this->getUserIp();
            $productName = $params['product_name'] ?? 'Ürün';
            $userBasket = htmlentities(json_encode([[($productName), $paymentAmount, 1]]));
            $paymentType = 'card';
            $currency = $params['currency'] ?? 'TL';
            $non3d = $params['non_3d'] ?? '0';
            $installmentCount = $params['installment_count'] ?? '0';
            $merchantOkUrl = $params['merchant_ok_url'];
            $merchantFailUrl = $params['merchant_fail_url'];
            $clientLang = $params['client_lang'] ?? 'tr';
            $userName = $params['buyer_name'] ?? $params['user_name'] ?? '';
            $userAddress = $params['buyer_address'] ?? $params['user_address'] ?? 'Türkiye';
            $userPhone = $params['buyer_phone'] ?? $params['user_phone'] ?? '05000000000';

            $hashStr = $this->merchantId . $userIp . $merchantOid . $email . $paymentAmount . $paymentType . $installmentCount . $currency . $this->testMode . $non3d;
            $token = base64_encode(hash_hmac('sha256', $hashStr . $this->merchantSalt, $this->merchantKey, true));

            return [
                'success' => true,
                'post_url' => 'https://www.paytr.com/odeme',
                'merchant_id' => $this->merchantId,
                'user_ip' => $userIp,
                'merchant_oid' => $merchantOid,
                'email' => $email,
                'payment_type' => $paymentType,
                'payment_amount' => $paymentAmount,
                'currency' => $currency,
                'test_mode' => $this->testMode,
                'non_3d' => $non3d,
                'merchant_ok_url' => $merchantOkUrl,
                'merchant_fail_url' => $merchantFailUrl,
                'user_basket' => $userBasket,
                'client_lang' => $clientLang,
                'paytr_token' => $token,
                'installment_count' => $installmentCount,
                'user_name' => $userName,
                'user_address' => $userAddress,
                'user_phone' => $userPhone,
                'non3d_test_failed' => '0',
            ];
        } catch (\Exception $e) {
            Log::error('PayTR payment creation failed', ['error' => $e->getMessage(), 'params' => $params]);
            return [
                'success' => false,
                'message' => 'Ödeme başlatılamadı: ' . $e->getMessage()
            ];
        }
    }

    public function verifyCallback($post)
    {
        if (!isset($post['merchant_oid']) || !isset($post['status']) || !isset($post['total_amount']) || !isset($post['hash'])) {
            return ['success' => false, 'message' => 'Eksik parametre'];
        }

        $hash = base64_encode(hash_hmac('sha256', $post['merchant_oid'] . $this->merchantSalt . $post['status'] . $post['total_amount'], $this->merchantKey, true));

        if ($hash != $post['hash']) {
            Log::error('PayTR callback hash verification failed', ['received' => $post['hash'], 'calculated' => $hash]);
            return ['success' => false, 'message' => 'Hash doğrulama başarısız'];
        }

        return [
            'success' => $post['status'] == 'success',
            'merchant_oid' => $post['merchant_oid'],
            'status' => $post['status'],
            'total_amount' => $post['total_amount'],
            'failed_reason_code' => $post['failed_reason_code'] ?? null,
            'failed_reason_msg' => $post['failed_reason_msg'] ?? null
        ];
    }

    protected function getUserIp()
    {
        if (isset($_SERVER["HTTP_CLIENT_IP"])) {
            return $_SERVER["HTTP_CLIENT_IP"];
        } elseif (isset($_SERVER["HTTP_X_FORWARDED_FOR"])) {
            return $_SERVER["HTTP_X_FORWARDED_FOR"];
        } else {
            return $_SERVER["REMOTE_ADDR"] ?? '127.0.0.1';
        }
    }
}

