<?php

namespace App\Services;

use App\Models\User;
use App\Models\Permission;
use Illuminate\Support\Facades\Cache;

class PermissionService
{
    private const CACHE_TTL = 300; 

    public function getUserPermissions(User $user): array
    {
        $cacheKey = "user_permissions_{$user->id}";
        
        return Cache::remember($cacheKey, self::CACHE_TTL, function () use ($user) {
            return $this->buildPermissionMatrix($user);
        });
    }

    private function buildPermissionMatrix(User $user): array
    {
        $role = $user->role;

        if ($role === 'super_admin') {
            return $this->getSuperAdminPermissions();
        }

        $permissions = Permission::whereHas('roles', function ($query) use ($role) {
            $query->where('role', $role);
        })->get();

        $allowedModules = [];
        $allowedActions = [];
        $restrictions = [];

        foreach ($permissions as $permission) {
            $module = $permission->module;
            $action = $permission->action;

            if (!in_array($module, $allowedModules)) {
                $allowedModules[] = $module;
            }

            if (!isset($allowedActions[$module])) {
                $allowedActions[$module] = [];
            }

            if (!in_array($action, $allowedActions[$module])) {
                $allowedActions[$module][] = $action;
            }

            $pivot = $permission->roles()->where('role', $role)->first()?->pivot;
            if ($pivot && $pivot->restrictions) {
                $restrictions[$module] = $pivot->restrictions;
            }
        }

        $permissionMatrix = [
            'role' => $role,
            'allowed_modules' => $allowedModules,
            'allowed_actions' => $allowedActions,
            'restrictions' => $restrictions,
            'notification_preferences' => $this->getNotificationPreferences($user),
        ];

        return $this->applyRoleOverrides($role, $permissionMatrix);
    }

    private function getSuperAdminPermissions(): array
    {
        $allModules = Permission::distinct('module')->pluck('module')->toArray();
        $allActions = [];

        foreach ($allModules as $module) {
            $allActions[$module] = Permission::where('module', $module)
                ->pluck('action')
                ->unique()
                ->toArray();
        }

        return [
            'role' => 'super_admin',
            'allowed_modules' => $allModules,
            'allowed_actions' => $allActions,
            'restrictions' => [],
            'notification_preferences' => [
                'appointment_reminders' => true,
                'appointment_approved' => true,
                'appointment_cancelled' => true,
                'invoice_created' => true,
                'payment_reminder' => true,
                'system_announcements' => true,
            ],
        ];
    }

    private function applyRoleOverrides(string $role, array $matrix): array
    {
        switch ($role) {
            case 'beauty_salon':
                
                if (!in_array('appointments', $matrix['allowed_modules'])) {
                    $matrix['allowed_modules'][] = 'appointments';
                }
                if (!isset($matrix['allowed_actions']['appointments'])) {
                    $matrix['allowed_actions']['appointments'] = [];
                }
                if (!in_array('approve', $matrix['allowed_actions']['appointments'])) {
                    $matrix['allowed_actions']['appointments'][] = 'approve';
                }
                break;

            case 'personel':
                
                $matrix['restrictions']['appointments'] = [
                    'scope' => 'own',
                    'can_delete' => false,
                    'can_edit' => false,
                ];
                break;
        }

        return $matrix;
    }

    private function getNotificationPreferences(User $user): array
    {
        
        $settings = \App\Models\Setting::where('user_id', $user->id)
            ->where('group', 'notifications')
            ->pluck('value', 'key')
            ->toArray();

        return [
            'appointment_reminders' => $settings['appointment_reminders'] ?? true,
            'appointment_approved' => $settings['appointment_approved'] ?? true,
            'appointment_cancelled' => $settings['appointment_cancelled'] ?? true,
            'invoice_created' => $settings['invoice_created'] ?? true,
            'payment_reminder' => $settings['payment_reminder'] ?? true,
            'system_announcements' => $settings['system_announcements'] ?? true,
        ];
    }

    public function clearUserCache(int $userId): void
    {
        Cache::forget("user_permissions_{$userId}");
    }
}

