<?php

namespace App\Services;

use App\Models\Reservation;
use App\Models\Customer;
use App\Models\User;
use App\Models\CustomerLoyalty;
use App\Models\LoyaltyTier;
use App\Models\LoyaltyPointsHistory;
use App\Models\Campaign;
use App\Models\CampaignRedemption;
use App\Models\CustomerMembership;
use App\Models\MembershipPackage;
use App\Models\MembershipUsageHistory;
use App\Models\StaffPerformanceMetric;
use App\Models\StaffServiceTracking;
use App\Models\StaffCommission;
use App\Models\CustomerClvScore;
use App\Models\CustomerSegment;
use App\Models\PricingRule;
use App\Models\DynamicPrice;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class PremiumFeaturesIntegrationService
{
    /**
     * Randevu tamamlandığında tüm premium özellikleri entegre et
     */
    public function onAppointmentCompleted(Reservation $reservation)
    {
        try {
            DB::beginTransaction();

            // 1. Sadakat Programı - Puan verme
            $this->awardLoyaltyPoints($reservation);

            // 2. Personel Performans - Metrik güncelleme
            $this->updateStaffPerformance($reservation);

            // 3. Personel Hizmet Takibi
            $this->trackStaffServices($reservation);

            // 4. CLV Hesaplama
            $this->calculateCustomerClv($reservation);

            DB::commit();
            
            Log::info('Premium özellikler entegre edildi', [
                'reservation_id' => $reservation->id,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Premium özellik entegrasyonu hatası: ' . $e->getMessage(), [
                'reservation_id' => $reservation->id,
                'trace' => $e->getTraceAsString(),
            ]);
        }
    }

    /**
     * Randevu oluşturulurken premium özellikleri kontrol et ve uygula
     */
    public function onAppointmentCreated(Reservation $reservation, array $options = [])
    {
        try {
            // 1. Kampanya kontrolü
            $campaignDiscount = $this->applyCampaignDiscount($reservation, $options['campaign_id'] ?? null);

            // 2. Üyelik kontrolü
            $membershipDiscount = $this->applyMembershipDiscount($reservation);

            // 3. Akıllı Fiyatlandırma
            $smartPricingAdjustment = $this->applySmartPricing($reservation);

            // Toplam indirimleri hesapla
            $totalDiscount = $campaignDiscount + $membershipDiscount + $smartPricingAdjustment;

            if ($totalDiscount > 0) {
                $newPrice = max(0, $reservation->total_price - $totalDiscount);
                $reservation->update(['total_price' => $newPrice]);
            }

            Log::info('Randevu oluşturulurken premium özellikler uygulandı', [
                'reservation_id' => $reservation->id,
                'campaign_discount' => $campaignDiscount,
                'membership_discount' => $membershipDiscount,
                'smart_pricing_adjustment' => $smartPricingAdjustment,
            ]);
        } catch (\Exception $e) {
            Log::error('Randevu oluşturulurken premium özellik uygulama hatası: ' . $e->getMessage(), [
                'reservation_id' => $reservation->id,
                'trace' => $e->getTraceAsString(),
            ]);
        }
    }

    /**
     * Sadakat Programı - Puan verme
     */
    private function awardLoyaltyPoints(Reservation $reservation)
    {
        if (!$reservation->customer_id || !$reservation->beauty_salon_id) {
            return;
        }

        $salon = User::find($reservation->beauty_salon_id);
        if (!$salon) {
            return;
        }

        $customer = Customer::find($reservation->customer_id);
        if (!$customer) {
            return;
        }

        // Harcama başına puan (örn: 100₺ = 10 puan)
        $pointsPerAmount = 10; // 100₺ başına 10 puan
        $pointsEarned = floor(($reservation->total_price / 100) * $pointsPerAmount);

        if ($pointsEarned <= 0) {
            return;
        }

        // Müşteri sadakat kaydını bul veya oluştur
        $loyalty = CustomerLoyalty::firstOrCreate(
            [
                'customer_id' => $customer->id,
                'beauty_salon_id' => $reservation->beauty_salon_id,
            ],
            [
                'total_points' => 0,
                'current_points' => 0,
                'loyalty_tier_id' => null,
            ]
        );

        // Puanları ekle
        $loyalty->increment('total_points', $pointsEarned);
        $loyalty->increment('current_points', $pointsEarned);

        // Seviye kontrolü
        $this->checkLoyaltyTier($loyalty, $reservation->beauty_salon_id);

        // Puan geçmişi kaydet
        LoyaltyPointsHistory::create([
            'customer_loyalty_id' => $loyalty->id,
            'points' => $pointsEarned,
            'type' => 'earned',
            'description' => "Randevu #{$reservation->code} tamamlandı - {$reservation->total_price}₺ harcama",
            'reference_type' => 'reservation',
            'reference_id' => $reservation->id,
        ]);

        Log::info('Sadakat puanı verildi', [
            'reservation_id' => $reservation->id,
            'customer_id' => $customer->id,
            'points_earned' => $pointsEarned,
        ]);
    }

    /**
     * Sadakat seviyesi kontrolü
     */
    private function checkLoyaltyTier(CustomerLoyalty $loyalty, $salonId)
    {
        $tiers = LoyaltyTier::where('beauty_salon_id', $salonId)
            ->where('is_active', true)
            ->orderBy('points_required', 'desc')
            ->get();

        foreach ($tiers as $tier) {
            if ($loyalty->total_points >= $tier->points_required) {
                if ($loyalty->loyalty_tier_id != $tier->id) {
                    $loyalty->update(['loyalty_tier_id' => $tier->id]);
                    Log::info('Müşteri sadakat seviyesi güncellendi', [
                        'customer_id' => $loyalty->customer_id,
                        'tier_id' => $tier->id,
                    ]);
                }
                break;
            }
        }
    }

    /**
     * Kampanya indirimi uygula
     */
    private function applyCampaignDiscount(Reservation $reservation, $campaignId = null): float
    {
        if (!$reservation->beauty_salon_id) {
            return 0;
        }

        $salon = User::find($reservation->beauty_salon_id);
        if (!$salon) {
            return 0;
        }

        $discount = 0;

        if ($campaignId) {
            $campaign = Campaign::where('id', $campaignId)
                ->where('beauty_salon_id', $reservation->beauty_salon_id)
                ->where('status', 'active')
                ->where('start_date', '<=', now())
                ->where('end_date', '>=', now())
                ->first();

            if ($campaign) {
                // Kampanya kurallarını kontrol et ve indirimi hesapla
                $discount = $this->calculateCampaignDiscount($campaign, $reservation);
            }
        }

        return $discount;
    }

    /**
     * Kampanya indirimi hesapla
     */
    private function calculateCampaignDiscount(Campaign $campaign, Reservation $reservation): float
    {
        $discount = 0;

        // Yüzde indirimi
        if ($campaign->discount_percentage > 0) {
            $discount = ($reservation->total_price * $campaign->discount_percentage) / 100;
        }

        // Sabit tutar indirimi
        if ($campaign->discount_amount > 0) {
            $discount = max($discount, min($campaign->discount_amount, $reservation->total_price));
        }

        // Kampanya kurallarını kontrol et
        if ($campaign->target_services && is_array($campaign->target_services)) {
            $hasTargetService = $reservation->services()->whereIn('services.id', $campaign->target_services)->exists();
            if (!$hasTargetService) {
                return 0; // Hedef hizmet yoksa indirim yok
            }
        }

        // Bütçe kontrolü
        if ($campaign->budget && $campaign->budget_used >= $campaign->budget) {
            return 0; // Bütçe dolmuş
        }

        // Maksimum kullanım kontrolü
        if ($campaign->max_redemptions && $campaign->redemptions_count >= $campaign->max_redemptions) {
            return 0; // Maksimum kullanım sayısına ulaşılmış
        }

        // Kampanya kullanımını kaydet
        CampaignRedemption::create([
            'campaign_id' => $campaign->id,
            'reservation_id' => $reservation->id,
            'customer_id' => $reservation->customer_id,
            'discount_amount' => $discount,
            'redeemed_at' => now(),
        ]);

        // Kampanya istatistiklerini güncelle
        $campaign->increment('redemptions_count');
        $campaign->increment('budget_used', $discount);

        return $discount;
    }

    /**
     * Üyelik indirimi uygula
     */
    private function applyMembershipDiscount(Reservation $reservation): float
    {
        if (!$reservation->customer_id || !$reservation->beauty_salon_id) {
            return 0;
        }

        $salon = User::find($reservation->beauty_salon_id);
        if (!$salon) {
            return 0;
        }

        $customer = Customer::find($reservation->customer_id);
        if (!$customer) {
            return 0;
        }

        $membership = CustomerMembership::where('customer_id', $customer->id)
            ->where('beauty_salon_id', $reservation->beauty_salon_id)
            ->where('status', 'active')
            ->where('expires_at', '>=', now())
            ->first();

        if (!$membership) {
            return 0;
        }

        $package = MembershipPackage::find($membership->membership_package_id);
        if (!$package || !$package->is_active) {
            return 0;
        }

        // Üyelik paketi kontrolü - Hizmet limiti ve tutar limiti kontrolü
        $discount = 0;
        
        // Kullanım geçmişini kontrol et
        $usageHistory = MembershipUsageHistory::where('customer_membership_id', $membership->id)
            ->where('created_at', '>=', $membership->start_date ?? $membership->created_at ?? now()->subYear())
            ->get();

        // Hizmet limiti kontrolü
        if ($package->service_limit && $usageHistory->count() >= $package->service_limit) {
            return 0; // Hizmet limiti dolmuş
        }

        // Tutar limiti kontrolü
        $totalUsed = $usageHistory->sum('amount_used');
        if ($package->amount_limit && $totalUsed >= $package->amount_limit) {
            return 0; // Tutar limiti dolmuş
        }

        // Üyelik paketi fiyatından indirim (paket fiyatına göre)
        // Burada üyelik paketinin kendisi bir avantaj olabilir, indirim değil
        // Eğer üyelik paketi belirli hizmetleri kapsıyorsa, o hizmetler ücretsiz veya indirimli olabilir
        // Şimdilik basit bir mantık: Üyelik paketi varsa %10 indirim (örnek)
        $discount = ($reservation->total_price * 10) / 100; // %10 indirim (bu değer ayarlanabilir)

        // Kullanım geçmişi kaydet
        MembershipUsageHistory::create([
            'customer_membership_id' => $membership->id,
            'reservation_id' => $reservation->id,
            'service_id' => $reservation->services()->first()?->id,
            'amount_used' => $reservation->total_price,
            'discount_applied' => $discount,
        ]);

        return $discount;
    }

    /**
     * Akıllı Fiyatlandırma uygula
     */
    private function applySmartPricing(Reservation $reservation): float
    {
        if (!$reservation->beauty_salon_id) {
            return 0;
        }

        $salon = User::find($reservation->beauty_salon_id);
        if (!$salon) {
            return 0;
        }

        $adjustment = 0;

        // Aktif fiyatlandırma kurallarını al
        $rules = PricingRule::where('beauty_salon_id', $reservation->beauty_salon_id)
            ->where('is_active', true)
            ->orderBy('priority', 'desc')
            ->get();

        foreach ($rules as $rule) {
            if ($this->isRuleApplicable($rule, $reservation)) {
                $adjustment += $this->calculateRuleAdjustment($rule, $reservation);
            }
        }

        return $adjustment;
    }

    /**
     * Kural uygulanabilir mi?
     */
    private function isRuleApplicable(PricingRule $rule, Reservation $reservation): bool
    {
        // Hizmet kontrolü
        if ($rule->service_id && !$reservation->services()->where('services.id', $rule->service_id)->exists()) {
            return false;
        }

        // Zaman kontrolü
        if ($rule->rule_type === 'time_based') {
            if ($rule->start_time && $rule->end_time) {
                $reservationTime = Carbon::parse($reservation->start_time);
                $ruleStart = Carbon::parse($rule->start_time);
                $ruleEnd = Carbon::parse($rule->end_time);
                
                if ($reservationTime->lt($ruleStart) || $reservationTime->gt($ruleEnd)) {
                    return false;
                }
            }
        }

        // Haftanın günü kontrolü
        if ($rule->days_of_week && is_array($rule->days_of_week)) {
            $dayOfWeek = Carbon::parse($reservation->date)->dayOfWeek;
            if (!in_array($dayOfWeek, $rule->days_of_week)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Kural ayarlamasını hesapla
     */
    private function calculateRuleAdjustment(PricingRule $rule, Reservation $reservation): float
    {
        if ($rule->pricing_type === 'percentage') {
            return ($reservation->total_price * $rule->price_adjustment) / 100;
        } elseif ($rule->pricing_type === 'fixed') {
            return $rule->price_adjustment;
        } elseif ($rule->pricing_type === 'multiplier') {
            $newPrice = $reservation->total_price * ($rule->multiplier ?? 1);
            return $newPrice - $reservation->total_price;
        }

        return 0;
    }

    /**
     * Personel Performans Güncelleme
     */
    private function updateStaffPerformance(Reservation $reservation)
    {
        if (!$reservation->personel_id || !$reservation->beauty_salon_id) {
            return;
        }

        $salon = User::find($reservation->beauty_salon_id);
        if (!$salon) {
            return;
        }

        $periodDate = Carbon::parse($reservation->date)->format('Y-m');

        // Performans metriklerini bul veya oluştur
        $metric = StaffPerformanceMetric::firstOrCreate(
            [
                'staff_id' => $reservation->personel_id,
                'beauty_salon_id' => $reservation->beauty_salon_id,
                'period_date' => $periodDate,
            ],
            [
                'total_appointments' => 0,
                'completed_appointments' => 0,
                'cancelled_appointments' => 0,
                'total_revenue' => 0,
                'average_rating' => 0,
                'total_rating_count' => 0,
            ]
        );

        // Metrikleri güncelle
        $metric->increment('total_appointments');
        $metric->increment('completed_appointments');
        $metric->increment('total_revenue', $reservation->total_price);

        // Komisyon hesapla
        $commission = $this->calculateStaffCommission($reservation);
        if ($commission > 0) {
            $metric->increment('total_commission', $commission);
        }

        $metric->save();

        Log::info('Personel performans güncellendi', [
            'reservation_id' => $reservation->id,
            'staff_id' => $reservation->personel_id,
        ]);
    }

    /**
     * Personel komisyon hesapla
     */
    private function calculateStaffCommission(Reservation $reservation): float
    {
        if (!$reservation->personel_id) {
            return 0;
        }

        $commission = StaffCommission::where('staff_id', $reservation->personel_id)
            ->where('beauty_salon_id', $reservation->beauty_salon_id)
            ->where('is_active', true)
            ->first();

        if (!$commission) {
            return 0;
        }

        if ($commission->commission_type === 'percentage') {
            return ($reservation->total_price * ($commission->commission_percentage ?? 0)) / 100;
        } elseif ($commission->commission_type === 'fixed') {
            return $commission->commission_amount ?? 0;
        }

        return 0;
    }

    /**
     * Personel Hizmet Takibi
     */
    private function trackStaffServices(Reservation $reservation)
    {
        if (!$reservation->personel_id || !$reservation->beauty_salon_id) {
            return;
        }

        $services = $reservation->services;
        if ($services->isEmpty()) {
            return;
        }

        foreach ($services as $service) {
            $commission = $this->calculateStaffCommission($reservation);
            
            StaffServiceTracking::create([
                'reservation_id' => $reservation->id,
                'staff_id' => $reservation->personel_id,
                'service_id' => $service->id,
                'beauty_salon_id' => $reservation->beauty_salon_id,
                'service_price' => $service->pivot->price ?? $service->price ?? 0,
                'commission_amount' => $commission / $services->count(), // Her hizmete eşit dağıt
                'service_date' => $reservation->date,
                'service_time' => $reservation->start_time,
                'status' => 'completed',
            ]);
        }

        Log::info('Personel hizmet takibi kaydedildi', [
            'reservation_id' => $reservation->id,
            'staff_id' => $reservation->personel_id,
        ]);
    }

    /**
     * CLV Hesaplama
     */
    private function calculateCustomerClv(Reservation $reservation)
    {
        if (!$reservation->customer_id || !$reservation->beauty_salon_id) {
            return;
        }

        $salon = User::find($reservation->beauty_salon_id);
        if (!$salon) {
            return;
        }

        $customer = Customer::find($reservation->customer_id);
        if (!$customer) {
            return;
        }

        // Randevu verilerini al
        $appointments = Reservation::where('customer_id', $customer->id)
            ->where('beauty_salon_id', $reservation->beauty_salon_id)
            ->where('status', 'completed')
            ->get();

        $totalSpend = $appointments->sum('total_price') ?? 0;
        $totalVisits = $appointments->count();
        $averageOrderValue = $totalVisits > 0 ? $totalSpend / $totalVisits : 0;
        
        // İlk ve son ziyaret tarihleri
        $firstVisit = $appointments->min('date');
        $lastVisit = $appointments->max('date');
        
        $daysSinceFirstVisit = $firstVisit ? now()->diffInDays($firstVisit) : 0;
        $daysSinceLastVisit = $lastVisit ? now()->diffInDays($lastVisit) : 0;
        
        // Purchase frequency (ayda kaç kez)
        $monthsActive = max(1, $daysSinceFirstVisit / 30);
        $purchaseFrequency = $monthsActive > 0 ? $totalVisits / $monthsActive : 0;
        
        // CLV = Average Order Value × Purchase Frequency × Customer Lifespan (months)
        $customerLifespan = max(1, $monthsActive);
        $clvScore = $averageOrderValue * $purchaseFrequency * $customerLifespan;
        
        // Churn risk hesaplama
        $churnRiskScore = 0;
        if ($daysSinceLastVisit > 90) {
            $churnRiskScore = min(100, 50 + ($daysSinceLastVisit - 90) / 2);
        } elseif ($daysSinceLastVisit > 60) {
            $churnRiskScore = 30;
        } elseif ($daysSinceLastVisit > 30) {
            $churnRiskScore = 15;
        }
        
        $churnRiskLevel = 'low';
        if ($churnRiskScore >= 70) {
            $churnRiskLevel = 'critical';
        } elseif ($churnRiskScore >= 50) {
            $churnRiskLevel = 'high';
        } elseif ($churnRiskScore >= 30) {
            $churnRiskLevel = 'medium';
        }
        
        // VIP kontrolü
        $isVip = $clvScore > 1000 || $totalVisits >= 10;
        
        // Segment belirleme
        $segment = null;
        if ($clvScore > 5000) {
            $segment = CustomerSegment::where('beauty_salon_id', $reservation->beauty_salon_id)
                ->where('slug', 'yuksek-clv')
                ->first();
        } elseif ($clvScore > 1000) {
            $segment = CustomerSegment::where('beauty_salon_id', $reservation->beauty_salon_id)
                ->where('slug', 'orta-clv')
                ->first();
        } else {
            $segment = CustomerSegment::where('beauty_salon_id', $reservation->beauty_salon_id)
                ->where('slug', 'dusuk-clv')
                ->first();
        }
        
        // CLV skorunu kaydet veya güncelle
        CustomerClvScore::updateOrCreate(
            [
                'customer_id' => $customer->id,
                'beauty_salon_id' => $reservation->beauty_salon_id,
            ],
            [
                'segment_id' => $segment?->id,
                'clv_score' => $clvScore,
                'average_order_value' => $averageOrderValue,
                'purchase_frequency' => $purchaseFrequency,
                'total_visits' => $totalVisits,
                'total_spend' => $totalSpend,
                'days_since_first_visit' => $daysSinceFirstVisit,
                'days_since_last_visit' => $daysSinceLastVisit,
                'churn_risk_score' => $churnRiskScore,
                'churn_risk_level' => $churnRiskLevel,
                'is_vip' => $isVip,
                'last_calculated_at' => now(),
            ]
        );

        Log::info('CLV hesaplandı', [
            'reservation_id' => $reservation->id,
            'customer_id' => $customer->id,
            'clv_score' => $clvScore,
        ]);
    }
}

