<?php

namespace App\Services\Sms;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\RateLimiter;

class OtpService
{
    protected $prefix;
    protected $length;
    protected $expiry;

    public function __construct()
    {
        $this->prefix = config('sms.otp.prefix', 'otp:');
        $this->length = config('sms.otp.length', 6);
        $this->expiry = config('sms.otp.expiry', 300); 
    }

    public function generate(): int
    {
        $min = pow(10, $this->length - 1);
        $max = pow(10, $this->length) - 1;
        return rand($min, $max);
    }

    public function store(string $phone, int $code, string $purpose = 'default'): void
    {
        $key = $this->getKey($phone, $purpose);
        Cache::put($key, $code, now()->addSeconds($this->expiry));
    }

    public function verify(string $phone, int $code, string $purpose = 'default'): bool
    {
        $key = $this->getKey($phone, $purpose);
        $storedCode = Cache::get($key);

        if ($storedCode === null) {
            return false;
        }

        if ((int)$storedCode === (int)$code) {
            Cache::forget($key);
            return true;
        }

        return false;
    }

    public function exists(string $phone, string $purpose = 'default'): bool
    {
        $key = $this->getKey($phone, $purpose);
        return Cache::has($key);
    }

    public function checkRateLimit(string $phone): bool
    {
        if (!config('sms.rate_limit.enabled', true)) {
            return true;
        }

        $maxPerMinute = config('sms.rate_limit.max_attempts_per_minute', 3);
        $key = "otp:{$phone}:minute";
        
        $attempts = RateLimiter::attempts($key);
        return $attempts < $maxPerMinute;
    }

    public function hitRateLimit(string $phone): void
    {
        $key = "otp:{$phone}:minute";
        RateLimiter::hit($key, 60); 
    }

    protected function getKey(string $phone, string $purpose): string
    {
        $normalizedPhone = $this->normalizePhone($phone);
        return "{$this->prefix}{$normalizedPhone}:{$purpose}";
    }

    protected function normalizePhone(string $phone): string
    {
        $phone = preg_replace('/[^0-9+]/', '', $phone);

        if (str_starts_with($phone, '0')) {
            $phone = '+90' . substr($phone, 1);
        } elseif (!str_starts_with($phone, '+')) {
            if (strlen($phone) === 10) {
                $phone = '+90' . $phone;
            } else {
                $phone = '+90' . $phone;
            }
        }
        
        return $phone;
    }
}

