<?php

namespace App\Services;

use App\DTO\SmsSendResult;
use App\Services\Sms\OtpService;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

class SmsService
{
    protected $otpService;
    protected $defaultProvider;

    public function __construct(OtpService $otpService)
    {
        $this->otpService = $otpService;
        $this->defaultProvider = config('sms.default', '1sms');
    }

    public function send(string $phone, string $message, ?string $provider = null, array $options = []): SmsSendResult
    {
        $provider = $provider ?? $this->defaultProvider;
        
        $phone = $this->normalizePhoneForSms($phone, $provider);
        $message = $this->normalizeTurkishMessage($message);
        $message = $this->normalizeMessageForProvider($message, $provider);

        $skipBalanceDeduction = $options['skip_balance_deduction'] ?? false;
        $userId = $options['user_id'] ?? null;

        if (!$skipBalanceDeduction && $userId) {
            $user = \App\Models\User::find($userId);
            if ($user && !$user->isSuperAdmin() && !$user->hasSmsBalance(1)) {
                return new SmsSendResult(false, null, 'SMS bakiyeniz yetersiz.');
            }
        }

        try {
            $result = $this->sendViaProvider($phone, $message, $provider);

            if ($result->success && !$skipBalanceDeduction && $userId) {
                $user = \App\Models\User::find($userId);
                if ($user) {
                    $user->deductSmsBalance(1);
                }
            }

            $this->logSms($phone, $message, $result, $options);
            
            return $result;
        } catch (\Exception $e) {
            Log::error('SMS gönderme hatası', [
                'phone' => $phone,
                'provider' => $provider,
                'error' => $e->getMessage()
            ]);

            return new SmsSendResult(false, null, $e->getMessage());
        }
    }


    public function sendOtp(string $phone, string $purpose = 'default', ?string $provider = null): array
    {
        Log::info('sendOtp başlatıldı', [
            'phone' => $phone,
            'purpose' => $purpose,
            'provider' => $provider
        ]);

        $normalizedPhone = $this->normalizePhone($phone);
        
        Log::info('Telefon normalize edildi', [
            'original' => $phone,
            'normalized' => $normalizedPhone
        ]);

        if (!$this->otpService->checkRateLimit($normalizedPhone)) {
            Log::warning('Rate limit aşıldı', ['phone' => $normalizedPhone]);
            return [
                'success' => false,
                'error' => 'rate_limit_exceeded',
                'message' => 'Çok fazla istek gönderdiniz. Lütfen bir süre bekleyin.'
            ];
        }

        $code = $this->otpService->generate();
        
        Log::info('OTP oluşturuldu', [
            'code' => $code,
            'phone' => $normalizedPhone,
            'purpose' => $purpose
        ]);

        $this->otpService->store($normalizedPhone, $code, $purpose);

        $this->otpService->hitRateLimit($normalizedPhone);

        $template = config('sms.otp.message_template', 'Doğrulama kodunuz: {code}');
        $message = str_replace('{code}', $code, $template);

        Log::info('SMS mesajı hazırlandı', [
            'message' => $message,
            'template' => $template
        ]);

        Log::info('OTP SMS üzerinden gönderiliyor', [
            'phone' => $phone,
            'purpose' => $purpose,
            'provider' => $this->defaultProvider
        ]);
        
        $provider = $this->defaultProvider;
        
        $result = $this->send($normalizedPhone, $message, $provider, [
            'skip_balance_deduction' => true,
            'purpose' => 'otp'
        ]);

        $message = $result->success ? 'OTP başarıyla gönderildi.' : ($result->error ?? 'OTP gönderilemedi.');
        
        // SMS API hatası varsa daha açıklayıcı mesaj
        if (!$result->success && $result->error) {
            // 1SMS API hatası kontrolü
            if (strpos($result->error, '1SMS API Hatası') !== false) {
                $message = 'SMS servisi şu anda kullanılamıyor. Lütfen daha sonra tekrar deneyin veya destek ekibi ile iletişime geçin.';
            }
        }
        
        return [
            'success' => $result->success,
            'code' => $code,
            'message' => $message,
            'error' => $result->success ? null : $result->error
        ];
    }

    public function verifyOtp(string $phone, int $code, string $purpose = 'default'): bool
    {
        
        $normalizedPhone = $this->normalizePhone($phone);
        return $this->otpService->verify($normalizedPhone, $code, $purpose);
    }

    protected function sendViaProvider(string $phone, string $message, string $provider): SmsSendResult
    {
        $config = config("sms.providers.{$provider}");
        
        if (!$config) {
            throw new \Exception("SMS provider '{$provider}' bulunamadı.");
        }

        $baseUrl = $config['base_url'] ?? '';
        $endpoint = $config['endpoints']['send'] ?? '/api/send';

        $endpoint = ltrim($endpoint, '/');
        $url = rtrim($baseUrl, '/') . '/' . $endpoint;

        if ($provider === '1sms') {
            return $this->sendVia1Sms($phone, $message, $config, $url);
        }

        $response = Http::timeout($config['timeout'] ?? 30)
            ->post($url, $this->buildPayload($phone, $message, $config));

        if ($response->successful()) {
            $data = $response->json();
            $messageId = $data['message_id'] ?? $data['id'] ?? null;
            return new SmsSendResult(true, $messageId);
        }

        return new SmsSendResult(false, null, $response->body());
    }

    protected function sendVia1Sms(string $phone, string $message, array $config, string $url): SmsSendResult
    {
        $username = $config['auth']['username'] ?? '';
        $password = $config['auth']['password'] ?? '';
        $sender = $config['sender'] ?? 'SALONAY';

        if (empty($username) || empty($password)) {
            Log::error('1SMS API: Kullanıcı adı veya şifre boş', [
                'username_set' => !empty($username),
                'password_set' => !empty($password)
            ]);
            return new SmsSendResult(false, null, '1SMS API kullanıcı adı veya şifre ayarlanmamış.');
        }

        // Telefon numarasını 1SMS formatına normalize et (90 ile başlamalı)
        $phone = $this->normalizePhoneForSms($phone, '1sms');
        
        // Validity değerini config'ten al, yoksa varsayılan 2880 (48 saat)
        $validity = $config['validity'] ?? 2880;

        // 1SMS API istegini UTF-8 XML bildirimi ile gonder.
        // Bu sayede Turkce karakterler (ü, ç, ş, ğ, İ, ö, ı) korunur.
        $xml = '<?xml version="1.0" encoding="UTF-8"?>';
        $xml .= '<sms>';
        $xml .= '<username>' . htmlspecialchars($username, ENT_XML1, 'UTF-8') . '</username>';
        $xml .= '<password>' . htmlspecialchars($password, ENT_XML1, 'UTF-8') . '</password>';
        $xml .= '<header>' . htmlspecialchars($sender, ENT_XML1, 'UTF-8') . '</header>';
        $xml .= '<validity>' . (int)$validity . '</validity>';
        $xml .= '<message>';
        $xml .= '<gsm>';
        $xml .= '<no>' . htmlspecialchars($phone, ENT_XML1, 'UTF-8') . '</no>';
        $xml .= '</gsm>';
        // CDATA kapanis dizisini guvenli hale getir.
        $xml .= '<msg><![CDATA[' . str_replace(']]>', ']]]]><![CDATA[>', $message) . ']]></msg>';
        $xml .= '</message>';
        $xml .= '</sms>';

        try {
            Log::info('1SMS API Request', [
                'url' => $url,
                'phone' => $phone,
                'phone_length' => strlen($phone),
                'sender' => $sender,
                'message' => $message,
                'message_length' => strlen($message),
                'message_length_chars' => mb_strlen($message, 'UTF-8'),
                'username' => $username ? 'SET' : 'NOT SET',
                'password' => $password ? 'SET' : 'NOT SET',
                'xml' => $xml
            ]);

            $response = Http::timeout($config['timeout'] ?? 30)
                ->withOptions([
                    // Sağlayıcı bazen hatalı Content-Encoding dönebiliyor (cURL 61).
                    // Otomatik decode'u kapatıp ham yanıtı alıyoruz.
                    'decode_content' => false,
                ])
                ->withHeaders([
                    'Content-Type' => 'text/xml; charset=UTF-8',
                    // Hatalı sıkıştırma/encoding pazarlığını önlemek için identity iste.
                    'Accept-Encoding' => 'identity',
                ])
                ->withBody($xml, 'text/xml; charset=UTF-8')
                ->post($url);

            $statusCode = $response->status();
            $responseBody = trim($response->body());

            Log::info('1SMS API Response', [
                'url' => $url,
                'status' => $statusCode,
                'body' => $responseBody
            ]);

            // 1SMS API başarılı dönüş: "00 77300600" veya "00 100420176 1770239123537" formatında (00 + boşluk + dlrId veya dlrId'ler)
            if ($statusCode === 200 && preg_match('/^00\s+(.+)$/', $responseBody, $matches)) {
                // İlk dlrId'yi al (birden fazla varsa ilkini kullan)
                $dlrIds = trim($matches[1]);
                $dlrId = explode(' ', $dlrIds)[0]; // İlk dlrId'yi al
                
                Log::info('1SMS API Başarılı', [
                    'phone' => $phone,
                    'dlrId' => $dlrId,
                    'allDlrIds' => $dlrIds,
                    'url' => $url,
                    'response' => $responseBody
                ]);
                return new SmsSendResult(true, $dlrId);
            }

            // Hata kodları: 2 basamaklı sayılar (87, 89, 85, vb.)
            if (preg_match('/^(\d{2})$/', $responseBody, $matches)) {
                $errorCode = (int)$matches[1];
                $errorMessage = $this->get1SmsErrorMessage($errorCode);
                
                Log::error('1SMS API Hata Kodu', [
                    'error_code' => $errorCode,
                    'error_message' => $errorMessage,
                    'phone' => $phone,
                    'xml' => $xml,
                    'response' => $responseBody
                ]);
                return new SmsSendResult(false, null, "1SMS API Hatası (Kod: {$errorCode}): {$errorMessage}");
            }

            // Eğer response "00" ile başlıyorsa ama regex eşleşmediyse, yine de başarılı sayalım
            if ($statusCode === 200 && strpos($responseBody, '00') === 0) {
                // "00" ile başlıyorsa başarılı kabul et
                $parts = explode(' ', $responseBody);
                $dlrId = isset($parts[1]) ? $parts[1] : uniqid('sms_', true);
                
                Log::info('1SMS API Başarılı (Alternatif Format)', [
                    'phone' => $phone,
                    'dlrId' => $dlrId,
                    'response' => $responseBody,
                    'url' => $url
                ]);
                return new SmsSendResult(true, $dlrId);
            }

            Log::error('1SMS API Hatası', [
                'status' => $statusCode,
                'body' => $responseBody,
                'phone' => $phone,
                'xml' => $xml
            ]);

            return new SmsSendResult(false, null, "1SMS API Hatası (HTTP {$statusCode}): " . $responseBody);

        } catch (\Exception $e) {
            Log::error('1SMS API Exception', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'phone' => $phone
            ]);
            return new SmsSendResult(false, null, $e->getMessage());
        }
    }

    protected function get1SmsErrorMessage(int $code): string
    {
        // 1SMS API dokümantasyonuna göre hata kodları
        $errorMessages = [
            // Başarısız Dönüşler
            87 => 'Kullanıcı adı veya şifre hatalı (WRONG_USER_OR_PASSWORD)',
            89 => 'XML formatı hatalı (WRONG_XML_FORMAT)',
            85 => 'Mesaj başlığı bulunamadı veya onaylanmamış (WRONG_SMS_HEADER)',
            84 => 'İleri tarihli gönderim zamanı hatalı (WRONG_SEND_DATE_TIME)',
            83 => 'Mesaj metni ve numaralar yetersiz (EMPTY_SMS)',
            81 => 'Yetersiz kredi (NOT_ENOUGH_CREDITS)',
            77 => 'Son 2 dakika içinde aynı SMS gönderilmiş (DUPLICATED_MESSAGE)',
            97 => 'HTTP POST metodu kullanılmalı (USE_POST_METHOD)',
            95 => 'HTTP GET metodu kullanılmalı (USE_GET_METHOD)',
            93 => 'GET parametrelerinde eksik var (MISSING_GET_PARAMS)',
            91 => 'POST verisi okunamadı veya yok (MISSING_POST_DATA)',
            99 => 'Bilinmeyen hata (UNKNOWN_ERROR)',
            // DLR hataları
            79 => 'DLR ID bulunamadı (DLR_ID_NOT_FOUND)',
            29 => 'Mesaj henüz yollanmamış (MESSAGE_WAITING_TO_SEND)',
            27 => 'Mesaj yollanırken hata oluştu (MESSAGE_SEND_ERROR)',
        ];
        
        return $errorMessages[$code] ?? "Bilinmeyen hata kodu: {$code}";
    }

    protected function buildPayload(string $phone, string $message, array $config): array
    {
        return [
            'phone' => $phone,
            'message' => $message,
            'sender' => $config['sender'] ?? 'SALONAY',
        ];
    }

    protected function normalizePhoneForSms(string $phone, string $provider): string
    {
        
        $phone = preg_replace('/[^0-9+]/', '', $phone);

        if ($provider === '1sms') {
            if (str_starts_with($phone, '0')) {
                $phone = '90' . substr($phone, 1);
            } elseif (str_starts_with($phone, '+90')) {
                $phone = '90' . substr($phone, 3);
            } elseif (!str_starts_with($phone, '90')) {
                
                if (strlen($phone) === 10) {
                    $phone = '90' . $phone;
                } else {
                    $phone = '90' . $phone;
                }
            }
            return $phone;
        }

        if (str_starts_with($phone, '0')) {
            $phone = '+90' . substr($phone, 1);
        } elseif (!str_starts_with($phone, '+')) {
            if (strlen($phone) === 10) {
                $phone = '+90' . $phone;
            } else {
                $phone = '+90' . $phone;
            }
        }
        
        return $phone;
    }

    protected function normalizePhone(string $phone): string
    {
        $phone = preg_replace('/[^0-9+]/', '', $phone);

        if (str_starts_with($phone, '0')) {
            $phone = '+90' . substr($phone, 1);
        } elseif (str_starts_with($phone, '90')) {
            $phone = '+90' . substr($phone, 2);
        } elseif (!str_starts_with($phone, '+')) {
            if (strlen($phone) === 10) {
                $phone = '+90' . $phone;
            } else {
                $phone = '+90' . $phone;
            }
        }
        
        return $phone;
    }

    /**
     * Tüm SMS kanallarında Türkçe karakterli metni merkezi olarak korur.
     * Özellikle geçmişte ASCII yazılmış sabit metinleri gönderim öncesi düzeltir.
     */
    protected function normalizeTurkishMessage(string $message): string
    {
        $message = trim($message);
        if ($message === '') {
            return $message;
        }

        // Güvenli UTF-8'e normalize et.
        if (!mb_check_encoding($message, 'UTF-8')) {
            $message = mb_convert_encoding($message, 'UTF-8', 'auto');
        }

        $replacements = [
            'Dogrulama' => 'Doğrulama',
            'dogrulama' => 'doğrulama',
            'dogrulamasina' => 'doğrulamasına',
            'gonderildi' => 'gönderildi',
            'Gonderildi' => 'Gönderildi',
            'gonderilirken' => 'gönderilirken',
            'gonderilemedi' => 'gönderilemedi',
            'Gonderilemedi' => 'Gönderilemedi',
            'onaylandi' => 'onaylandı',
            'Onaylandi' => 'Onaylandı',
            'onaylanmistir' => 'onaylanmıştır',
            'Onaylanmistir' => 'Onaylanmıştır',
            'tamamlandi' => 'tamamlandı',
            'Tamamlandi' => 'Tamamlandı',
            'tamamlanmistir' => 'tamamlanmıştır',
            'olusturulmustur' => 'oluşturulmuştur',
            'Olusturulmustur' => 'Oluşturulmuştur',
            'musteri' => 'müşteri',
            'Musteri' => 'Müşteri',
            'Kayitli' => 'Kayıtlı',
            'geciliyor' => 'geçiliyor',
            'Lutfen' => 'Lütfen',
            'tamamlayin' => 'tamamlayın',
            'Sifre' => 'Şifre',
            'sifre' => 'şifre',
            'odeme' => 'ödeme',
            'Odeme' => 'Ödeme',
            'gunler' => 'günler',
            'diler' => 'diler',
            'linkiniz' => 'linkiniz',
            'Fatura linkiniz' => 'Fatura linkiniz',
            'Sozlesme' => 'Sözleşme',
            'saatinizden' => 'saatinizden',
            'bulunmanizi' => 'bulunmanızı',
            'rica ederiz' => 'rica ederiz',
            'ulasabilirsiniz' => 'ulaşabilirsiniz',
            'oylamak' => 'oylamak',
            'asagidaki' => 'aşağıdaki',
            'Referans' => 'Referans',
            'Kodunuz' => 'Kodunuz',
            'arkadaslarinizla' => 'arkadaşlarınızla',
            'paylasin' => 'paylaşın',
            'kazandirir' => 'kazandırır',
            'Faturaniz' => 'Faturanız',
            'kodunuz' => 'kodunuz',
            'Doğrulama kodunuz' => 'Doğrulama kodunuz',
        ];

        return strtr($message, $replacements);
    }

    /**
     * Sağlayıcıya giderken metni bozulmayan UTF-8 formatına getirir.
     */
    protected function normalizeMessageForProvider(string $message, string $provider): string
    {
        if ($message === '') {
            return $message;
        }

        // UTF-8 olmayan metinler Türkçe karakter kaybına neden olabilir.
        if (!mb_check_encoding($message, 'UTF-8')) {
            $message = mb_convert_encoding($message, 'UTF-8', 'auto');
        }

        // Kontrol karakterlerini temizle (satır sonu ve tab dışında).
        $message = preg_replace('/[^\P{C}\t\r\n]/u', '', $message) ?? $message;

        // 1SMS için BOM ve gizli karakter temizliği.
        if ($provider === '1sms') {
            $message = preg_replace('/^\xEF\xBB\xBF/u', '', $message) ?? $message;
        }

        return $message;
    }


    protected function logSms(string $phone, string $message, SmsSendResult $result, array $options = []): void
    {
        try {
            
            $phoneColumn = Schema::hasColumn('sms_logs', 'to_phone') ? 'to_phone' : 'to';
            
            $values = [
                $phoneColumn => $phone,
                'message' => $message,
                'status' => $result->success ? 'sent' : 'failed',
                'created_at' => now(),
                'updated_at' => now(),
            ];

            if (Schema::hasColumn('sms_logs', 'message_id')) {
                $values['message_id'] = $result->messageId;
            }
            if (Schema::hasColumn('sms_logs', 'error')) {
                $values['error'] = $result->error;
            }
            if (Schema::hasColumn('sms_logs', 'purpose')) {
                $values['purpose'] = $options['purpose'] ?? null;
            }
            if (Schema::hasColumn('sms_logs', 'user_id')) {
                $values['user_id'] = $options['user_id'] ?? null;
            }

            DB::table('sms_logs')->insert($values);
        } catch (\Exception $e) {
            Log::error('SMS log kaydetme hatası', ['error' => $e->getMessage()]);
        }
    }
}

