<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\SmsService;
use App\Services\Sms\OtpService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class AccountController extends Controller
{
    protected $smsService;
    protected $otpService;

    public function __construct(SmsService $smsService, OtpService $otpService)
    {
        $this->smsService = $smsService;
        $this->otpService = $otpService;
    }

    public function index()
    {
        $user = Auth::user();
        return view('admin.account.index', compact('user'));
    }

    /**
     * Send OTP for account deletion
     */
    public function sendDeleteOtp(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->phone) {
            return response()->json([
                'success' => false,
                'message' => 'Telefon numaranız kayıtlı değil. Hesap silme işlemi için telefon numaranızı güncelleyin.',
            ], 400);
        }

        // OTP gönder
        $result = $this->smsService->sendOtp($user->phone, 'account_deletion');

        if ($result['success']) {
            return response()->json([
                'success' => true,
                'message' => 'Doğrulama kodu telefon numaranıza gönderildi.',
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => $result['message'] ?? 'OTP gönderilemedi. Lütfen tekrar deneyin.',
        ], 400);
    }

    /**
     * Delete account with OTP verification
     */
    public function destroy(Request $request)
    {
        $request->validate([
            'otp_code' => 'required|string|size:6',
            'password' => 'required|string',
        ], [
            'otp_code.required' => 'Doğrulama kodu zorunludur.',
            'otp_code.size' => 'Doğrulama kodu 6 haneli olmalıdır.',
            'password.required' => 'Şifre zorunludur.',
        ]);

        $user = Auth::user();

        // Şifre kontrolü
        if (!Hash::check($request->password, $user->password)) {
            return back()->withErrors(['password' => 'Şifre hatalı.'])->withInput();
        }

        // OTP doğrulama
        $otpCode = (int) $request->otp_code;
        if (!$this->otpService->verify($user->phone, $otpCode, 'account_deletion')) {
            return back()->withErrors(['otp_code' => 'Doğrulama kodu hatalı veya süresi dolmuş.'])->withInput();
        }

        try {
            DB::beginTransaction();

            // Kullanıcıyı sil (ilişkili veriler cascade delete ile silinecek)
            $userName = $user->name;
            $user->delete();

            DB::commit();

            Auth::logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();

            return redirect()->route('home')->with('success', 'Hesabınız başarıyla silindi. Üzgünüz ki ayrılıyoruz...');
            
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Hesap silme hatası: ' . $e->getMessage());
            return back()->withErrors(['error' => 'Hesap silinirken bir hata oluştu. Lütfen tekrar deneyin.'])->withInput();
        }
    }

    public function update(Request $request)
    {
        $user = Auth::user();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'nullable|string',
            'password' => 'nullable|min:8|confirmed',
        ]);

        if ($request->filled('password')) {
            $validated['password'] = Hash::make($validated['password']);
        } else {
            unset($validated['password']);
        }

        $user->update($validated);

        return redirect()->route('admin.account.index')
            ->with('success', 'Profil bilgileri güncellendi.');
    }
}

