<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Account;
use App\Models\Transaction;
use App\Models\Customer;
use App\Models\Reservation;
use App\Models\ActivityLog;
use App\Models\Invoice;
use App\Models\Setting;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CashboxController extends Controller
{
    public function index(Request $request)
    {
        try {
            $user = Auth::user();
            
            // Kullanıcıya özel kasa hesabını getir
            $cashAccount = Account::getCashAccount($user->id);
            
            // Gelir/Gider hesaplamaları (data isolation ile) - DOĞRU HESAPLAMA
            $baseQuery = Transaction::where('account_id', $cashAccount->id);
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $baseQuery->where('created_by', $user->id);
            }
            
            $totalIncome = 0;
            $totalExpense = 0;
            $todayIncome = 0;
            $todayExpense = 0;
            $monthlyIncome = 0;
            $monthlyExpense = 0;
            
            try {
                // Toplam gelir/gider (kasa hesabına özel)
                $totalIncome = (clone $baseQuery)->where('type', 'income')->sum('amount') ?? 0;
                $totalExpense = (clone $baseQuery)->where('type', 'expense')->sum('amount') ?? 0;
                
                // Bugünkü gelir/gider
                $today = date('Y-m-d');
                $todayIncome = (clone $baseQuery)->where('type', 'income')->whereDate('date', $today)->sum('amount') ?? 0;
                $todayExpense = (clone $baseQuery)->where('type', 'expense')->whereDate('date', $today)->sum('amount') ?? 0;
                
                // Bu ayki gelir/gider
                $monthStart = date('Y-m-01');
                $monthEnd = date('Y-m-t');
                $monthlyIncome = (clone $baseQuery)->where('type', 'income')->whereBetween('date', [$monthStart, $monthEnd])->sum('amount') ?? 0;
                $monthlyExpense = (clone $baseQuery)->where('type', 'expense')->whereBetween('date', [$monthStart, $monthEnd])->sum('amount') ?? 0;
                
                // Kasa bakiyesi = Gelir - Gider (DOĞRU HESAPLAMA)
                $totalBalance = $totalIncome - $totalExpense;
            } catch (\Exception $e) {
                \Log::warning('Gelir/Gider hesaplanamadı: ' . $e->getMessage());
                $totalBalance = 0;
            }
            
            // Tüm gelir ve gider işlemlerini getir (kasa hesabına özel)
            $query = Transaction::where('account_id', $cashAccount->id)
                ->whereIn('type', ['income', 'expense']);
            
            // Rezervasyoncu ise sadece kendi işlemlerini göster
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
            
            // İlişkileri güvenli şekilde yükle
            try {
                $query->with(['account', 'reservation', 'customer']);
            } catch (\Exception $e) {
                \Log::warning('Transaction ilişkileri yüklenemedi: ' . $e->getMessage());
            }
            
            // Sıralama
            try {
                $query->orderBy('date', 'desc')->orderBy('created_at', 'desc');
            } catch (\Exception $e) {
                // date kolonu yoksa created_at'e göre sırala
                try {
                    $query->orderBy('created_at', 'desc');
                } catch (\Exception $e2) {
                    \Log::warning('Transaction sıralama hatası: ' . $e2->getMessage());
                }
            }
            
            // Filtreleme
            if ($request->filled('date_from')) {
                try {
                    $query->whereDate('date', '>=', $request->date_from);
                } catch (\Exception $e) {
                    \Log::warning('Tarih filtresi uygulanamadı: ' . $e->getMessage());
                }
            }
            if ($request->filled('date_to')) {
                try {
                    $query->whereDate('date', '<=', $request->date_to);
                } catch (\Exception $e) {
                    \Log::warning('Tarih filtresi uygulanamadı: ' . $e->getMessage());
                }
            }
            if ($request->filled('type') && in_array($request->type, ['income', 'expense'])) {
                $query->where('type', $request->type);
            }
            
            $transactions = $query->paginate(50);
            
            // Müşteriler ve rezervasyonlar (modal için) - Data isolation
            $customers = collect([]);
            $reservations = collect([]);
            
            try {
                $customerQuery = Customer::query();
                // Rezervasyoncu ise sadece kendi müşterilerini göster
                if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                    $customerQuery->where('created_by', $user->id);
                }
                $customers = $customerQuery->orderBy('name')->get();
            } catch (\Exception $e) {
                \Log::warning('Müşteriler yüklenemedi: ' . $e->getMessage());
            }
            
            try {
                $reservationQuery = Reservation::where('status', '!=', 'cancelled');
                // Rezervasyoncu ise sadece kendi rezervasyonlarını göster
                if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                    $reservationQuery->where('created_by', $user->id);
                }
                $reservationQuery->with('customer');
                try {
                    $reservationQuery->orderBy('date', 'desc');
                } catch (\Exception $e) {
                    $reservationQuery->orderBy('created_at', 'desc');
                }
                $reservations = $reservationQuery->limit(100)->get();
            } catch (\Exception $e) {
                \Log::warning('Rezervasyonlar yüklenemedi: ' . $e->getMessage());
                // date kolonu yoksa created_at'e göre dene
                try {
                    $reservationQuery = Reservation::where('status', '!=', 'cancelled');
                    // Rezervasyoncu ise sadece kendi rezervasyonlarını göster
                    if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                        $reservationQuery->where('created_by', $user->id);
                    }
                    $reservations = $reservationQuery->with('customer')
                        ->orderBy('created_at', 'desc')
                        ->limit(100)
                        ->get();
                } catch (\Exception $e2) {
                    \Log::warning('Rezervasyonlar yüklenemedi (fallback): ' . $e2->getMessage());
                }
            }
            
            return view('admin.accounting.cashbox', compact(
                'transactions', 
                'totalBalance', 
                'customers', 
                'reservations',
                'totalIncome',
                'totalExpense',
                'todayIncome',
                'todayExpense',
                'monthlyIncome',
                'monthlyExpense'
            ))->with('filters', [
                    'date_from' => $request->date_from,
                    'date_to' => $request->date_to,
                    'type' => $request->type,
                ]);
        } catch (\Exception $e) {
            \Log::error('Cashbox Index Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            
            // Fallback data
            return view('admin.accounting.cashbox', [
                'transactions' => \Illuminate\Pagination\Paginator::make([], 0, 50),
                'totalBalance' => 0,
                'totalIncome' => 0,
                'totalExpense' => 0,
                'todayIncome' => 0,
                'todayExpense' => 0,
                'monthlyIncome' => 0,
                'monthlyExpense' => 0,
                'customers' => collect([]),
                'reservations' => collect([]),
                'filters' => [
                    'date_from' => $request->date_from,
                    'date_to' => $request->date_to,
                    'type' => $request->type,
                ]
            ])->with('error', 'Kasa sayfası yüklenirken bir hata oluştu. Lütfen sayfayı yenileyin.');
        }
    }

    public function storeIncome(Request $request)
    {
        try {
            $validated = $request->validate([
                'amount' => 'required|numeric|min:0',
                'date' => 'required|date',
                'time' => 'nullable|string',
                'description' => 'nullable|string',
                'payer_name' => 'nullable|string|max:255',
                'customer_id' => 'nullable|exists:customers,id',
                'reservation_id' => 'nullable|exists:reservations,id',
            ]);
            
            // date kolonu sadece date tipinde, datetime değil
            // Tarih formatını düzelt (Y-m-d formatında olmalı)
            $validated['date'] = date('Y-m-d', strtotime($validated['date']));

            // Kullanıcıya özel kasa hesabını getir
            $cashAccount = Account::getCashAccount(Auth::id());

            // Açıklamaya ödeme yapan kişiyi ekle
            if (!empty($validated['payer_name'])) {
                $payerInfo = "Ödeme Yapan: " . $validated['payer_name'];
                if (!empty($validated['description'])) {
                    $validated['description'] = $payerInfo . " | " . $validated['description'];
                } else {
                    $validated['description'] = $payerInfo;
                }
            }

            $validated['account_id'] = $cashAccount->id;
            $validated['type'] = 'income';
            $validated['created_by'] = Auth::id();
        
        // Rezervasyon seçildiyse müşteri ID'sini de ekle - Data isolation kontrolü
        if (!empty($validated['reservation_id']) && empty($validated['customer_id'])) {
            $reservationQuery = Reservation::where('id', $validated['reservation_id']);
            // Rezervasyoncu ise sadece kendi rezervasyonunu seçebilir
            if (Auth::user()->role === 'rezervasyoncu' || Auth::user()->role === 'admin') {
                $reservationQuery->where('created_by', Auth::id());
            }
            $reservation = $reservationQuery->first();
            if ($reservation && $reservation->customer_id) {
                $validated['customer_id'] = $reservation->customer_id;
            } else if (!empty($validated['reservation_id'])) {
                // Rezervasyon bulunamadı veya yetki yok
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Seçilen rezervasyon bulunamadı veya yetkiniz yok.');
            }
        }
        
        // Müşteri seçildiyse data isolation kontrolü
        if (!empty($validated['customer_id'])) {
            $customerQuery = Customer::where('id', $validated['customer_id']);
            // Rezervasyoncu ise sadece kendi müşterisini seçebilir
            if (Auth::user()->role === 'rezervasyoncu' || Auth::user()->role === 'admin') {
                $customerQuery->where('created_by', Auth::id());
            }
            $customer = $customerQuery->first();
            if (!$customer) {
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Seçilen müşteri bulunamadı veya yetkiniz yok.');
            }
        }

        $transaction = Transaction::create($validated);

        // Müşteri bakiyesini güncelle (gelir = müşteriden borç düşer, negatif olabilir)
        if (!empty($validated['customer_id'])) {
            $customer = Customer::find($validated['customer_id']);
            if ($customer) {
                // Müşteri bakiyesinden borç düş (decrement)
                // Eğer fazla gelir eklenirse negatife düşebilir (müşteriye borçlu oluruz)
                $customer->decrement('balance', $validated['amount']);
            }
        }

        // Rezervasyon seçildiyse, rezervasyon kalan bakiyesini güncelle (gelir = kalan bakiye azalır)
        if (!empty($validated['reservation_id'])) {
            $reservationQuery = Reservation::where('id', $validated['reservation_id']);
            // Rezervasyoncu ise sadece kendi rezervasyonunu güncelleyebilir
            if (Auth::user()->role === 'rezervasyoncu' || Auth::user()->role === 'admin') {
                $reservationQuery->where('created_by', Auth::id());
            }
            $reservation = $reservationQuery->first();
            if ($reservation && isset($reservation->remaining_amount) && $reservation->remaining_amount > 0) {
                $reservation->decrement('remaining_amount', min($validated['amount'], $reservation->remaining_amount));
            }
        }

            // ActivityLog oluştur (hata olsa bile devam et)
            try {
                ActivityLog::create([
                    'user_id' => Auth::id(),
                    'action' => 'create_income',
                    'description' => "Gelir eklendi: {$validated['amount']} TL",
                    'ip_address' => $request->ip(),
                ]);
            } catch (\Exception $e) {
                \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
            }

            // Otomatik fatura oluştur (rezervasyon varsa veya yoksa) - HER ZAMAN ZORUNLU
            try {
                \Log::info('Fatura oluşturma başlatılıyor (Gelir)', [
                    'transaction_id' => $transaction->id,
                    'reservation_id' => $validated['reservation_id'] ?? null,
                    'amount' => $validated['amount'],
                    'type' => $validated['type']
                ]);
                
                // Transaction'ı refresh et
                $transaction->refresh();
                
                if (!empty($validated['reservation_id'])) {
                    // Rezervasyon varsa rezervasyon için fatura oluştur
                    $invoice = $this->createInvoiceForReservation($validated['reservation_id']);
                    if (!$invoice) {
                        throw new \Exception('Rezervasyon için fatura oluşturulamadı');
                    }
                    \Log::info('Rezervasyon için fatura oluşturuldu (Gelir)', ['invoice_id' => $invoice->id]);
                } else {
                    // Rezervasyon yoksa transaction için fatura oluştur - HER ZAMAN ZORUNLU
                    $invoice = $this->createInvoiceForTransaction($transaction);
                    if (!$invoice) {
                        throw new \Exception('Transaction için fatura oluşturulamadı - createInvoiceForTransaction null döndü');
                    }
                    \Log::info('Transaction için fatura oluşturuldu (Gelir)', [
                        'invoice_id' => $invoice->id,
                        'invoice_number' => $invoice->invoice_number,
                        'transaction_id' => $transaction->id
                    ]);
                }
            } catch (\Exception $e) {
                \Log::error('Fatura oluşturma hatası (Gelir): ' . $e->getMessage(), [
                    'transaction_id' => $transaction->id,
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'trace' => $e->getTraceAsString()
                ]);
                // Hata olsa bile devam et ama logla - kullanıcıya uyarı göster
            }

            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Gelir başarıyla eklendi.',
                    'transaction' => $transaction
                ]);
            }

            return redirect()->route('admin.accounting.cashbox')
                ->with('success', 'Gelir başarıyla eklendi.');
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation hatası',
                    'errors' => $e->errors()
                ], 422);
            }
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            \Log::error('Store Income Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Gelir eklenirken bir hata oluştu: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', 'Gelir eklenirken bir hata oluştu: ' . $e->getMessage());
        }
    }

    public function storeExpense(Request $request)
    {
        try {
            $validated = $request->validate([
                'amount' => 'required|numeric|min:0',
                'date' => 'required|date',
                'time' => 'nullable|string',
                'description' => 'nullable|string',
                'customer_id' => 'nullable|exists:customers,id',
                'reservation_id' => 'nullable|exists:reservations,id',
            ]);
            
            // date kolonu sadece date tipinde, datetime değil
            // Tarih formatını düzelt (Y-m-d formatında olmalı)
            $validated['date'] = date('Y-m-d', strtotime($validated['date']));

            // Kullanıcıya özel kasa hesabını getir
            $cashAccount = Account::getCashAccount(Auth::id());

            $validated['account_id'] = $cashAccount->id;
            $validated['type'] = 'expense';
            $validated['created_by'] = Auth::id();
            
            // Rezervasyon seçildiyse müşteri ID'sini de ekle - Data isolation kontrolü
            if (!empty($validated['reservation_id']) && (empty($validated['customer_id']) || $validated['customer_id'] == '')) {
                $reservationQuery = Reservation::where('id', $validated['reservation_id']);
                // Rezervasyoncu ise sadece kendi rezervasyonunu seçebilir
                if (Auth::user()->role === 'rezervasyoncu' || Auth::user()->role === 'admin') {
                    $reservationQuery->where('created_by', Auth::id());
                }
                $reservation = $reservationQuery->first();
                if ($reservation && $reservation->customer_id) {
                    $validated['customer_id'] = $reservation->customer_id;
                } else if (!empty($validated['reservation_id'])) {
                    // Rezervasyon bulunamadı veya yetki yok
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Seçilen rezervasyon bulunamadı veya yetkiniz yok.');
                }
            }
            
            // Müşteri seçildiyse data isolation kontrolü
            if (!empty($validated['customer_id']) && $validated['customer_id'] != '') {
                $customerQuery = Customer::where('id', $validated['customer_id']);
                // Rezervasyoncu ise sadece kendi müşterisini seçebilir
                if (Auth::user()->role === 'rezervasyoncu' || Auth::user()->role === 'admin') {
                    $customerQuery->where('created_by', Auth::id());
                }
                $customer = $customerQuery->first();
                if (!$customer) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Seçilen müşteri bulunamadı veya yetkiniz yok.');
                }
            }

            $transaction = Transaction::create($validated);

            // Müşteri bakiyesini güncelle (gider = müşteriye borç olarak eklenir)
            if (!empty($validated['customer_id']) && $validated['customer_id'] != '') {
                $customer = Customer::find($validated['customer_id']);
                if ($customer) {
                    $customer->increment('balance', $validated['amount']);
                }
            }

            // Rezervasyon seçildiyse, rezervasyon kalan bakiyesini güncelle (gider = kalan bakiye artar)
            if (!empty($validated['reservation_id'])) {
                $reservationQuery = Reservation::where('id', $validated['reservation_id']);
                // Rezervasyoncu ise sadece kendi rezervasyonunu güncelleyebilir
                if (Auth::user()->role === 'rezervasyoncu' || Auth::user()->role === 'admin') {
                    $reservationQuery->where('created_by', Auth::id());
                }
                $reservation = $reservationQuery->first();
                if ($reservation && isset($reservation->remaining_amount)) {
                    $reservation->increment('remaining_amount', $validated['amount']);
                }
            }

            // ActivityLog oluştur (hata olsa bile devam et)
            try {
                ActivityLog::create([
                    'user_id' => Auth::id(),
                    'action' => 'create_expense',
                    'description' => "Gider eklendi: {$validated['amount']} TL",
                    'ip_address' => $request->ip(),
                ]);
            } catch (\Exception $e) {
                \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
            }

            // Otomatik fatura oluştur (rezervasyon varsa veya yoksa) - HER ZAMAN ZORUNLU
            try {
                \Log::info('Fatura oluşturma başlatılıyor (Gider)', [
                    'transaction_id' => $transaction->id,
                    'reservation_id' => $validated['reservation_id'] ?? null,
                    'amount' => $validated['amount'],
                    'type' => $validated['type']
                ]);
                
                // Transaction'ı refresh et
                $transaction->refresh();
                
                if (!empty($validated['reservation_id'])) {
                    // Rezervasyon varsa rezervasyon için fatura oluştur
                    $invoice = $this->createInvoiceForReservation($validated['reservation_id']);
                    if (!$invoice) {
                        throw new \Exception('Rezervasyon için fatura oluşturulamadı');
                    }
                    \Log::info('Rezervasyon için fatura oluşturuldu (Gider)', ['invoice_id' => $invoice->id]);
                } else {
                    // Rezervasyon yoksa transaction için fatura oluştur - HER ZAMAN ZORUNLU
                    $invoice = $this->createInvoiceForTransaction($transaction);
                    if (!$invoice) {
                        throw new \Exception('Transaction için fatura oluşturulamadı - createInvoiceForTransaction null döndü');
                    }
                    \Log::info('Transaction için fatura oluşturuldu (Gider)', [
                        'invoice_id' => $invoice->id,
                        'invoice_number' => $invoice->invoice_number,
                        'transaction_id' => $transaction->id
                    ]);
                }
            } catch (\Exception $e) {
                \Log::error('Fatura oluşturma hatası (Gider): ' . $e->getMessage(), [
                    'transaction_id' => $transaction->id,
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'trace' => $e->getTraceAsString()
                ]);
                // Hata olsa bile devam et ama logla - kullanıcıya uyarı göster
            }

            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Gider başarıyla eklendi.',
                    'transaction' => $transaction
                ]);
            }

            return redirect()->route('admin.accounting.cashbox')
                ->with('success', 'Gider başarıyla eklendi.');
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation hatası',
                    'errors' => $e->errors()
                ], 422);
            }
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            \Log::error('Store Expense Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Gider eklenirken bir hata oluştu: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', 'Gider eklenirken bir hata oluştu: ' . $e->getMessage());
        }
    }

    public function show($id)
    {
        try {
            $transaction = Transaction::with(['account', 'customer', 'reservation'])->findOrFail($id);
            
            // Data isolation kontrolü
            $user = Auth::user();
            if (($user->role === 'rezervasyoncu' || $user->role === 'admin') && $transaction->created_by != $user->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bu işlemi görüntüleme yetkiniz yok.'
                ], 403);
            }
            
            return response()->json([
                'success' => true,
                'transaction' => [
                    'id' => $transaction->id,
                    'amount' => $transaction->amount,
                    'date' => $transaction->date ? $transaction->date->format('Y-m-d') : ($transaction->created_at ? $transaction->created_at->format('Y-m-d') : ''),
                    'description' => $transaction->description,
                    'customer_id' => $transaction->customer_id,
                    'type' => $transaction->type,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'İşlem bulunamadı: ' . $e->getMessage()
            ], 404);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $transaction = Transaction::with(['account', 'customer', 'reservation'])->findOrFail($id);
            
            // Data isolation kontrolü
            $user = Auth::user();
            if (($user->role === 'rezervasyoncu' || $user->role === 'admin') && $transaction->created_by != $user->id) {
                return redirect()->back()->with('error', 'Bu işlemi düzenleme yetkiniz yok.');
            }
            
            $oldAmount = $transaction->amount;
            $oldCustomerId = $transaction->customer_id;
            $oldType = $transaction->type;
            
            $validated = $request->validate([
                'amount' => 'required|numeric|min:0',
                'date' => 'required|date',
                'description' => 'nullable|string',
                'customer_id' => 'nullable|exists:customers,id',
            ]);
            
            $validated['date'] = date('Y-m-d', strtotime($validated['date']));
            
            // Müşteri seçildiyse data isolation kontrolü
            if (!empty($validated['customer_id'])) {
                $customerQuery = Customer::where('id', $validated['customer_id']);
                if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                    $customerQuery->where('created_by', $user->id);
                }
                $customer = $customerQuery->first();
                if (!$customer) {
                    return redirect()->back()
                        ->withInput()
                        ->with('error', 'Seçilen müşteri bulunamadı veya yetkiniz yok.');
                }
            }
            
            // Eski müşteri bakiyesini geri al
            if ($oldCustomerId) {
                $oldCustomer = Customer::find($oldCustomerId);
                if ($oldCustomer) {
                    if ($oldType === 'income') {
                        $oldCustomer->increment('balance', $oldAmount);
                    } else {
                        $oldCustomer->decrement('balance', $oldAmount);
                    }
                }
            }
            
            // Yeni değerleri güncelle
            $transaction->update($validated);
            
            // Yeni müşteri bakiyesini güncelle
            if (!empty($validated['customer_id'])) {
                $customer = Customer::find($validated['customer_id']);
                if ($customer) {
                    if ($transaction->type === 'income') {
                        $customer->decrement('balance', $validated['amount']);
                    } else {
                        $customer->increment('balance', $validated['amount']);
                    }
                }
            }
            
            // ActivityLog
            try {
                ActivityLog::create([
                    'user_id' => Auth::id(),
                    'action' => 'update_transaction',
                    'description' => "İşlem güncellendi: {$transaction->type} - {$validated['amount']} TL",
                    'ip_address' => $request->ip(),
                ]);
            } catch (\Exception $e) {
                \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
            }
            
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'İşlem başarıyla güncellendi.',
                    'transaction' => $transaction
                ]);
            }
            
            return redirect()->route('admin.accounting.cashbox')
                ->with('success', 'İşlem başarıyla güncellendi.');
        } catch (\Exception $e) {
            \Log::error('Update Transaction Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'İşlem güncellenirken bir hata oluştu: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', 'İşlem güncellenirken bir hata oluştu: ' . $e->getMessage());
        }
    }

    public function destroy($id)
    {
        $transaction = Transaction::with(['account', 'customer', 'reservation'])->findOrFail($id);
        
        // Data isolation kontrolü
        $user = Auth::user();
        if (($user->role === 'rezervasyoncu' || $user->role === 'admin') && $transaction->created_by != $user->id) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bu işlemi silme yetkiniz yok.'
                ], 403);
            }
            return redirect()->back()->with('error', 'Bu işlemi silme yetkiniz yok.');
        }
        
        $amount = $transaction->amount;
        $type = $transaction->type;
        
        // İşlemi geri al - müşteri ve rezervasyon bakiyelerini düzelt (kasa otomatik hesaplanır)
        if ($type === 'income') {
            // Müşteri bakiyesini düzelt (gelir silinince müşteri bakiyesi artar - borç artar)
            if ($transaction->customer_id) {
                $customer = Customer::find($transaction->customer_id);
                if ($customer) {
                    $customer->increment('balance', $amount);
                }
            }
            
            // Rezervasyon kalan bakiyesini düzelt
            if ($transaction->reservation_id) {
                $reservation = Reservation::find($transaction->reservation_id);
                if ($reservation) {
                    $newRemaining = min($reservation->remaining_amount + $amount, $reservation->total_price);
                    $reservation->update(['remaining_amount' => $newRemaining]);
                }
            }
        } else {
            // Müşteri bakiyesini düzelt (gider silinince müşteri bakiyesi azalır)
            if ($transaction->customer_id) {
                $customer = Customer::find($transaction->customer_id);
                if ($customer) {
                    $customer->decrement('balance', min($amount, $customer->balance));
                }
            }
            
            // Rezervasyon kalan bakiyesini düzelt
            if ($transaction->reservation_id) {
                $reservation = Reservation::find($transaction->reservation_id);
                if ($reservation) {
                    $newRemaining = max(0, $reservation->remaining_amount - $amount);
                    $reservation->update(['remaining_amount' => $newRemaining]);
                }
            }
        }
        
        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'delete_transaction',
            'description' => "İşlem silindi: {$type} - {$amount} TL",
            'ip_address' => request()->ip(),
        ]);
        
        $transaction->delete();
        
        if (request()->ajax() || request()->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'İşlem başarıyla silindi.',
            ]);
        }
        
        return redirect()->route('admin.accounting.cashbox')
            ->with('success', 'İşlem başarıyla silindi.');
    }
    
    /**
     * Kasa raporu - JSON API
     */
    public function report(Request $request)
    {
        try {
            $user = Auth::user();
            $startDate = $request->get('start_date', now()->startOfMonth()->format('Y-m-d'));
            $endDate = $request->get('end_date', now()->format('Y-m-d'));
            
            $query = Transaction::query();
            
            // Data isolation
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
            
            // Tarih filtresi
            $query->whereBetween('date', [$startDate, $endDate]);
            
            // Özet hesaplamalar
            $totalIncome = (clone $query)->where('type', 'income')->sum('amount');
            $totalExpense = (clone $query)->where('type', 'expense')->sum('amount');
            $netBalance = $totalIncome - $totalExpense;
            
            // Günlük bazda breakdown
            $dailyBreakdown = (clone $query)
                ->selectRaw('DATE(date) as day, type, SUM(amount) as total')
                ->groupBy('day', 'type')
                ->orderBy('day')
                ->get()
                ->groupBy('day')
                ->map(function($items, $day) {
                    return [
                        'date' => $day,
                        'income' => $items->where('type', 'income')->sum('total'),
                        'expense' => $items->where('type', 'expense')->sum('total'),
                    ];
                })->values();
            
            // Müşteri bazlı breakdown
            $customerBreakdown = (clone $query)
                ->with('customer:id,name')
                ->whereNotNull('customer_id')
                ->selectRaw('customer_id, type, SUM(amount) as total')
                ->groupBy('customer_id', 'type')
                ->get()
                ->groupBy('customer_id')
                ->map(function($items) {
                    $customer = $items->first()->customer;
                    return [
                        'customer_id' => $items->first()->customer_id,
                        'customer_name' => $customer ? $customer->name : 'Bilinmeyen',
                        'income' => $items->where('type', 'income')->sum('total'),
                        'expense' => $items->where('type', 'expense')->sum('total'),
                    ];
                })->values();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'period' => [
                        'start' => $startDate,
                        'end' => $endDate,
                    ],
                    'summary' => [
                        'total_income' => $totalIncome,
                        'total_expense' => $totalExpense,
                        'net_balance' => $netBalance,
                    ],
                    'daily_breakdown' => $dailyBreakdown,
                    'customer_breakdown' => $customerBreakdown,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Rapor oluşturulurken hata: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Kasa bakiyesini yeniden hesapla (kullanıcıya özel)
     */
    public function recalculate()
    {
        try {
            $user = Auth::user();
            $cashAccount = Account::getCashAccount($user->id);
            $newBalance = $cashAccount->recalculateBalance();
            
            return response()->json([
                'success' => true,
                'message' => 'Kasa bakiyesi yeniden hesaplandı.',
                'balance' => $newBalance
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Hata: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Rezervasyon için fatura oluştur
     */
    private function createInvoiceForReservation($reservationId)
    {
        // Rezervasyonu yükle
        $reservation = Reservation::with(['customer', 'salon'])->find($reservationId);
        if (!$reservation) {
            return null;
        }
        
        // Fatura numarası oluştur
        $invoiceNumber = 'FAT-' . date('Y') . '-' . str_pad(Invoice::count() + 1, 6, '0', STR_PAD_LEFT);
        
        // Public token oluştur
        $publicToken = Str::random(32);
        
        // Ayarları al
        $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
        $logo = $settings['invoice_logo'] ?? null;
        $address = $reservation->salon->address ?? ($settings['invoice_address'] ?? '');
        $taxOffice = $settings['invoice_tax_office'] ?? '';
        $taxNumber = $settings['invoice_tax_number'] ?? '';
        
        // Fatura verilerini hazırla (ReservationController'daki prepareInvoiceData metodunu kullan)
        $reservationController = new \App\Http\Controllers\Admin\ReservationController();
        $reflection = new \ReflectionClass($reservationController);
        $method = $reflection->getMethod('prepareInvoiceData');
        $method->setAccessible(true);
        $invoiceData = $method->invoke($reservationController, $reservation, $logo, $address, $taxOffice, $taxNumber);
        
        // PDF oluştur
        $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);
        
        // PDF'i kaydet
        $pdfPath = 'invoices/' . $invoiceNumber . '.pdf';
        Storage::disk('public')->put($pdfPath, $pdf->output());
        
        // Public URL oluştur
        $publicUrl = 'https://adugun.com/fatura/' . $publicToken;
        
        // Faturayı veritabanına kaydet
        $invoice = Invoice::create([
            'reservation_id' => $reservation->id,
            'customer_id' => $reservation->customer_id,
            'invoice_number' => $invoiceNumber,
            'total_amount' => $reservation->total_price,
            'deposit_paid' => $reservation->deposit_paid ?? 0,
            'remaining_amount' => $reservation->remaining_amount ?? ($reservation->total_price - ($reservation->deposit_paid ?? 0)),
            'invoice_date' => now(),
            'pdf_path' => $pdfPath,
            'public_token' => $publicToken,
            'public_url' => $publicUrl,
            'status' => 'sent',
            'items_json' => $invoiceData['items'],
            'created_by' => Auth::id(),
        ]);
        
        return $invoice;
    }
    
    /**
     * Transaction için fatura oluştur (rezervasyon formatında)
     */
    /**
     * Transaction için fatura oluştur (Rezervasyon sisteminden kopyalandı)
     */
    private function createInvoiceForTransaction($transaction)
    {
        // Transaction'ı yükle
        $transaction = Transaction::with(['customer', 'account'])->find($transaction->id);
        if (!$transaction) {
            \Log::error('Transaction bulunamadı', ['transaction_id' => $transaction->id ?? 'null']);
            return null;
        }
        
        // Fatura numarası oluştur
        $invoiceNumber = 'FAT-' . date('Y') . '-' . str_pad(Invoice::count() + 1, 6, '0', STR_PAD_LEFT);
        
        // Public token oluştur
        $publicToken = Str::random(32);
        
        // Transaction açıklamasını hazırla
        $itemName = $transaction->type === 'income' ? 'Gelir İşlemi' : 'Gider İşlemi';
        $itemDescription = $transaction->description ?? $itemName;
        
        // Fatura kalemlerini hazırla (rezervasyon sistemindeki gibi)
        // Transaction_id'yi item içine ekle (fatura bulmak için)
        $items = [[
            'name' => $itemDescription,
            'quantity' => '-',
            'price' => $transaction->amount,
            'total' => $transaction->amount,
            '_transaction_id' => $transaction->id, // Fatura bulmak için
        ]];
        
        // Public URL oluştur
        $publicUrl = 'https://adugun.com/fatura/' . $publicToken;
        
        // Faturayı veritabanına kaydet (rezervasyon sistemindeki gibi)
        \Log::info('📝 Fatura oluşturuluyor', [
            'transaction_id' => $transaction->id,
            'invoice_number' => $invoiceNumber,
            'amount' => $transaction->amount,
            'items_json' => $items
        ]);
        
        $invoice = Invoice::create([
            'reservation_id' => null, // Rezervasyon yok
            'customer_id' => $transaction->customer_id,
            'invoice_number' => $invoiceNumber,
            'total_amount' => $transaction->amount,
            'deposit_paid' => $transaction->type === 'income' ? $transaction->amount : 0,
            'remaining_amount' => 0,
            'invoice_date' => $transaction->date ? \Carbon\Carbon::parse($transaction->date) : now(),
            'pdf_path' => null, // PDF oluşturulmayacak
            'public_token' => $publicToken,
            'public_url' => $publicUrl,
            'status' => 'sent',
            'items_json' => $items, // Direkt items array (rezervasyon sistemindeki gibi)
            'created_by' => Auth::id(),
        ]);
        
        \Log::info('✅ Fatura başarıyla oluşturuldu', [
            'invoice_id' => $invoice->id,
            'invoice_number' => $invoice->invoice_number,
            'transaction_id' => $transaction->id
        ]);
        
        return $invoice;
    }
    
    /**
     * Transaction'a ait faturayı bul ve göster
     */
    public function getInvoiceForTransaction($id)
    {
        try {
            $transaction = Transaction::findOrFail($id);
            
            // Data isolation kontrolü
            $user = Auth::user();
            if (($user->role === 'rezervasyoncu' || $user->role === 'admin') && $transaction->created_by != $user->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Bu işleme ait faturayı görüntüleme yetkiniz yok.'
                ], 403);
            }
            
            \Log::info('🔍 Fatura aranıyor', [
                'transaction_id' => $transaction->id,
                'amount' => $transaction->amount,
                'created_by' => $transaction->created_by,
                'date' => $transaction->date
            ]);
            
            // TÜM transaction faturalarını çek
            $allInvoices = Invoice::where('created_by', $transaction->created_by)
                ->whereNull('reservation_id')
                ->get();
            
            \Log::info('📋 Bulunan faturalar', [
                'count' => $allInvoices->count(),
                'invoice_ids' => $allInvoices->pluck('id')->toArray()
            ]);
            
            // Önce items_json içinde transaction_id'yi ara
            $invoice = $allInvoices->filter(function($inv) use ($transaction) {
                $itemsJson = is_array($inv->items_json) ? $inv->items_json : (is_string($inv->items_json) ? json_decode($inv->items_json, true) : []);
                
                // items_json bir array, ilk elemanı kontrol et
                if (is_array($itemsJson) && isset($itemsJson[0]) && is_array($itemsJson[0])) {
                    if (isset($itemsJson[0]['_transaction_id']) && $itemsJson[0]['_transaction_id'] == $transaction->id) {
                        \Log::info('✅ Fatura bulundu (transaction_id ile)', ['invoice_id' => $inv->id]);
                        return true;
                    }
                }
                return false;
            })->first();
            
            // Eğer bulunamazsa, tarih + tutar ile bul
            if (!$invoice) {
                $transactionCreatedAt = $transaction->created_at;
                $invoiceDate = $transaction->date ? $transaction->date->format('Y-m-d') : $transactionCreatedAt->format('Y-m-d');
                
                \Log::info('🔍 Tarih + tutar ile aranıyor', [
                    'invoice_date' => $invoiceDate,
                    'amount' => $transaction->amount,
                    'created_at' => $transactionCreatedAt->format('Y-m-d H:i:s')
                ]);
                
                $invoice = Invoice::where('created_by', $transaction->created_by)
                    ->whereNull('reservation_id')
                    ->where('total_amount', $transaction->amount)
                    ->whereDate('invoice_date', $invoiceDate)
                    ->where('created_at', '>=', $transactionCreatedAt->copy()->subMinutes(10))
                    ->where('created_at', '<=', $transactionCreatedAt->copy()->addHours(2))
                    ->orderBy('created_at', 'asc')
                    ->first();
                
                if ($invoice) {
                    \Log::info('✅ Fatura bulundu (tarih + tutar ile)', ['invoice_id' => $invoice->id]);
                }
            }
            
            // Hala bulunamazsa, sadece tutar ve kullanıcıya göre bul (transaction'dan sonra)
            if (!$invoice) {
                \Log::info('🔍 Sadece tutar ile aranıyor');
                
                $invoice = Invoice::where('created_by', $transaction->created_by)
                    ->whereNull('reservation_id')
                    ->where('total_amount', $transaction->amount)
                    ->where('created_at', '>=', $transaction->created_at->copy()->subMinutes(10))
                    ->where('created_at', '<=', $transaction->created_at->copy()->addHours(2))
                    ->orderBy('created_at', 'asc')
                    ->first();
                
                if ($invoice) {
                    \Log::info('✅ Fatura bulundu (sadece tutar ile)', ['invoice_id' => $invoice->id]);
                }
            }
            
            // Son çare: Sadece tutar (tarih filtresi yok)
            if (!$invoice) {
                \Log::info('🔍 Son çare: Sadece tutar (tarih yok)');
                
                $invoice = Invoice::where('created_by', $transaction->created_by)
                    ->whereNull('reservation_id')
                    ->where('total_amount', $transaction->amount)
                    ->orderBy('created_at', 'desc')
                    ->first();
                
                if ($invoice) {
                    \Log::info('✅ Fatura bulundu (son çare)', ['invoice_id' => $invoice->id]);
                }
            }
            
            if (!$invoice) {
                \Log::error('❌ Fatura bulunamadı', [
                    'transaction_id' => $transaction->id,
                    'amount' => $transaction->amount,
                    'created_by' => $transaction->created_by
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Bu işlem için fatura bulunamadı. Lütfen yeni bir işlem oluşturun.'
                ], 404);
            }
            
            \Log::info('✅ Fatura bulundu ve döndürülüyor', [
                'invoice_id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number
            ]);
            
            return response()->json([
                'success' => true,
                'invoice' => [
                    'id' => $invoice->id,
                    'invoice_number' => $invoice->invoice_number,
                    'public_url' => $invoice->public_url,
                    'total_amount' => $invoice->total_amount,
                    'invoice_date' => $invoice->invoice_date->format('d.m.Y'),
                ]
            ]);
        } catch (\Exception $e) {
            \Log::error('❌ Get Invoice For Transaction Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Fatura bulunurken bir hata oluştu: ' . $e->getMessage()
            ], 500);
        }
    }
}

