<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\Reservation;
use App\Models\ActivityLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;

class CustomerController extends Controller
{
    public function index(Request $request)
    {
        try {
            $user = Auth::user();
            
            // Basit sorgu - sadece müşterileri getir
            $query = Customer::query();
            
            // Rezervasyoncu ise sadece kendi müşterilerini göster
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }

            // Arama
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('phone', 'like', "%{$search}%");
                    
                    // Email varsa ara (kolon yoksa hata vermez)
                    try {
                        $q->orWhere('email', 'like', "%{$search}%");
                    } catch (\Exception $e) {
                        // Email kolonu yoksa devam et
                    }
                });
            }

            // Sıralama
            $sortBy = $request->get('sort', 'name_asc'); // Default: A-Z
            switch ($sortBy) {
                case 'name_asc':
                    $query->orderBy('name', 'asc');
                    break;
                case 'created_desc':
                    $query->orderBy('created_at', 'desc');
                    break;
                default:
                    $query->orderBy('name', 'asc');
            }

            // JSON format isteniyorsa
            if ($request->get('format') === 'json' || $request->wantsJson() || $request->ajax()) {
                $customers = $query->get(['id', 'name', 'phone', 'email']);
                return response()->json([
                    'success' => true,
                    'customers' => $customers
                ]);
            }
            
            $customers = $query->paginate(20);
            
            // Her müşteri için toplam kalan bakiye hesapla (doğru hesaplama)
            foreach ($customers as $customer) {
                try {
                    // Rezervasyonları getir
                    $reservationQuery = Reservation::where('customer_id', $customer->id);
                    if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                        $reservationQuery->where('created_by', $user->id);
                    }
                    try {
                        $reservationQuery->where('status', '!=', 'cancelled');
                    } catch (\Exception $e) {
                        // status kolonu yoksa devam et
                    }
                    $reservations = $reservationQuery->get();
                    
                    // Transaction'ları getir
                    $transactionQuery = \App\Models\Transaction::where('customer_id', $customer->id);
                    if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                        $transactionQuery->where('created_by', $user->id);
                    }
                    
                    // Toplam borç = rezervasyonların toplam fiyatı
                    $totalDebt = $reservations->sum('total_price') ?? 0;
                    
                    // Ödenen = deposit_paid + income transaction'ları
                    $totalDepositPaid = $reservations->sum('deposit_paid') ?? 0;
                    $totalIncomeTransactions = (clone $transactionQuery)->where('type', 'income')->sum('amount') ?? 0;
                    $totalPaid = $totalDepositPaid + $totalIncomeTransactions;
                    
                    // Kalan = Toplam borç - Ödenen (negatif olamaz)
                    $customer->total_remaining = max(0, $totalDebt - $totalPaid);
                } catch (\Exception $e) {
                    $customer->total_remaining = 0;
                    \Log::warning('Müşteri bakiye hesaplama hatası: ' . $e->getMessage());
                }
            }

            return view('admin.customers.index', compact('customers'));
        } catch (\Exception $e) {
            \Log::error('Customer Index Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            
            // Fallback data
            try {
                return view('admin.customers.index', [
                    'customers' => \Illuminate\Pagination\Paginator::make([], 0, 20)
                ])->with('error', 'Müşteriler yüklenirken bir hata oluştu: ' . $e->getMessage());
            } catch (\Exception $viewError) {
                \Log::error('View render error: ' . $viewError->getMessage());
                return response()->view('errors.500', ['message' => 'Müşteriler yüklenirken bir hata oluştu.'], 500);
            }
        }
    }

    public function create()
    {
        return view('admin.customers.create');
    }

    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'phone' => 'required|string|max:20',
                'email' => 'nullable|email|max:255',
                'address' => 'nullable|string',
                'notes' => 'nullable|string',
            ]);

            // Sadece mevcut kolonları kullan
            $customerData = [
                'name' => $validated['name'],
                'phone' => $validated['phone'],
            ];
            
            // created_by kolonu varsa ekle
            if (Schema::hasColumn('customers', 'created_by')) {
                $customerData['created_by'] = Auth::id();
            }
            
            // Opsiyonel alanları ekle (varsa)
            if (isset($validated['email']) && !empty($validated['email'])) {
                $customerData['email'] = $validated['email'];
            }
            if (isset($validated['address']) && !empty($validated['address'])) {
                $customerData['address'] = $validated['address'];
            }
            if (isset($validated['notes']) && !empty($validated['notes'])) {
                $customerData['notes'] = $validated['notes'];
            }
            
            $customer = Customer::create($customerData);

            // ActivityLog oluştur (hata olsa bile devam et)
            try {
                ActivityLog::create([
                    'user_id' => Auth::id(),
                    'action' => 'create_customer',
                    'description' => "Yeni müşteri eklendi: {$customer->name}",
                    'ip_address' => $request->ip(),
                ]);
            } catch (\Exception $e) {
                \Log::warning('ActivityLog kaydedilemedi: ' . $e->getMessage());
            }

            // AJAX isteği ise JSON döndür
            if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
                return response()->json([
                    'success' => true,
                    'message' => 'Müşteri başarıyla eklendi.',
                    'customer' => [
                        'id' => $customer->id,
                        'name' => $customer->name,
                        'phone' => $customer->phone ?? '',
                        'email' => $customer->email ?? '',
                    ]
                ], 200);
            }

            // Eğer redirect_back parametresi varsa geri dön
            if ($request->has('redirect_back')) {
                return redirect()->back()->with('success', 'Müşteri başarıyla eklendi: ' . $customer->name);
            }
            
            return redirect()->route('admin.customers.index')
                ->with('success', 'Müşteri başarıyla eklendi.');
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation hatası',
                    'errors' => $e->errors()
                ], 422);
            }
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            \Log::error('Customer Store Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            
            if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
                return response()->json([
                    'success' => false,
                    'message' => 'Müşteri oluşturulurken bir hata oluştu: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->back()
                ->withInput()
                ->with('error', 'Müşteri oluşturulurken bir hata oluştu: ' . $e->getMessage());
        }
    }

    public function search(Request $request)
    {
        $query = $request->get('q', '');
        
        $user = Auth::user();
        $customerQuery = Customer::query();
        
        // Rezervasyoncu ise sadece kendi müşterilerini göster
        if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
            $customerQuery->where('created_by', $user->id);
        }
        
        // Eğer query varsa filtrele, yoksa tümünü getir
        if (strlen($query) >= 2) {
            $customerQuery->where(function($q) use ($query) {
                $q->where('name', 'like', "%{$query}%")
                  ->orWhere('phone', 'like', "%{$query}%");
            });
        }
        
        $customers = $customerQuery
            ->orderBy('name')
            ->limit(20) // Maksimum 20 müşteri göster
            ->get()
            ->map(function($customer) {
                return [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'phone' => $customer->phone,
                    'email' => $customer->email ?? '',
                    'initials' => $customer->initials ?? '??',
                    'avatar_color' => $customer->avatar_color ?? '#2563eb',
                ];
            });
        
        return response()->json($customers);
    }

    public function show($id)
    {
        $user = Auth::user();
        $query = Customer::query();
        
        // Rezervasyoncu ise sadece kendi müşterisini görebilir
        if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $customer = $query->findOrFail($id);
        
        // Rezervasyonları filtrele - sadece bu rezervasyoncuya ait olanları getir
        $reservationQuery = Reservation::where('customer_id', $customer->id)
            ->with(['salon', 'menu']);
        
        if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
            $reservationQuery->where('created_by', $user->id);
        }
        
        $customer->reservations = $reservationQuery->get();

        // Müşterinin hesap bordosu (transactions) - en eski önce sırala (kalan borç hesaplaması için)
        // Rezervasyoncu ise sadece kendi transaction'larını göster
        $transactionsQuery = \App\Models\Transaction::where('customer_id', $customer->id);
        
        if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
            $transactionsQuery->where('created_by', $user->id);
        }

        // İstatistikleri doğru hesapla
        $totalReservations = $customer->reservations->count();
        $totalAmount = $customer->reservations->sum('total_price') ?? 0;
        
        // Ödenen tutar = deposit_paid + transaction'lardan gelen income'lar
        $totalPaid = $customer->reservations->sum('deposit_paid') ?? 0;
        $transactionPaid = (clone $transactionsQuery)->where('type', 'income')->sum('amount') ?? 0;
        $totalPaid = $totalPaid + $transactionPaid;
        
        // Kalan = Toplam - Ödenen
        $totalRemaining = max(0, $totalAmount - $totalPaid);
        
        $transactionsOrdered = $transactionsQuery->orderBy('date', 'asc')
            ->orderBy('created_at', 'asc')
            ->get();

        // Önce tüm rezervasyonların toplam borcunu hesapla
        $totalReservationDebt = $customer->reservations->sum('total_price');
        
        // Her transaction için kalan borcu hesapla
        $cumulativePaid = 0;
        foreach ($transactionsOrdered as $index => $transaction) {
            // Bu transaction'dan önceki tüm income transaction'ların toplamını hesapla
            $paidBeforeThis = $transactionsOrdered->take($index)->where('type', 'income')->sum('amount');
            
            // Bu transaction dahil ödenen toplam (sadece income transaction'lar için)
            if ($transaction->type == 'income') {
                $cumulativePaid = $paidBeforeThis + $transaction->amount;
                // Bu transaction'dan sonra kalan borç
                $transaction->remaining_debt_after = max(0, $totalReservationDebt - $cumulativePaid);
            } else {
                // Expense transaction'lar için önceki kalan borcu göster
                $transaction->remaining_debt_after = max(0, $totalReservationDebt - $paidBeforeThis);
            }
        }

        // View için ters sırala (en yeni önce)
        $transactions = $transactionsOrdered->reverse()->values();

        return view('admin.customers.show', compact('customer', 'totalReservations', 'totalAmount', 'totalPaid', 'totalRemaining', 'transactions'));
    }

    public function edit($id)
    {
        $user = Auth::user();
        $query = Customer::query();
        
        // Rezervasyoncu ise sadece kendi müşterisini düzenleyebilir
        if ($user->role === 'rezervasyoncu') {
            $query->where('created_by', $user->id);
        }
        
        $customer = $query->findOrFail($id);
        return view('admin.customers.edit', compact('customer'));
    }

    public function update(Request $request, $id)
    {
        $customer = Customer::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string',
            'notes' => 'nullable|string',
        ]);

        // Sadece mevcut kolonları güncelle
        $updateData = [
            'name' => $validated['name'],
            'phone' => $validated['phone'],
        ];
        
        // Opsiyonel alanları ekle (varsa)
        if (isset($validated['email'])) {
            $updateData['email'] = $validated['email'];
        }
        if (isset($validated['address'])) {
            $updateData['address'] = $validated['address'];
        }
        if (isset($validated['notes'])) {
            $updateData['notes'] = $validated['notes'];
        }
        
        $customer->update($updateData);

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'update_customer',
            'description' => "Müşteri güncellendi: {$customer->name}",
            'ip_address' => $request->ip(),
        ]);

        return redirect()->route('admin.customers.index')
            ->with('success', 'Müşteri başarıyla güncellendi.');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        $query = Customer::query();
        
        // Rezervasyoncu ise sadece kendi müşterisini silebilir
        if ($user->role === 'rezervasyoncu') {
            $query->where('created_by', $user->id);
        }
        
        $customer = $query->findOrFail($id);
        $name = $customer->name;
        $customer->delete();

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'delete_customer',
            'description' => "Müşteri silindi: {$name}",
            'ip_address' => request()->ip(),
        ]);

        return redirect()->route('admin.customers.index')
            ->with('success', 'Müşteri başarıyla silindi.');
    }
}

