<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Transaction;
use App\Models\Account;
use App\Models\Customer;
use App\Models\Salon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function index()
    {
        try {
            // Türkçe locale ve İstanbul timezone ayarla
            Carbon::setLocale('tr');
            date_default_timezone_set('Europe/Istanbul');
            
            $user = auth()->user();
            
            // Süper admin ise super admin dashboard'a yönlendir
            if ($user->isSuperAdmin()) {
                return redirect()->route('admin.superadmin.dashboard');
            }
            
            // Kullanıcının salonu var mı kontrol et (ilk kurulum için)
            $hasSalon = Salon::where('created_by', $user->id)->exists();
            $showSetupWizard = !$hasSalon;
            
            $today = now()->setTimezone('Europe/Istanbul')->format('Y-m-d');
            
            // Kasa bilgileri
            $cashBalance = 0;
            $cashIncome = 0;
            $cashExpense = 0;
            $todayIncome = 0;
            $todayExpense = 0;
            $pendingPayments = 0;
            
            try {
                // Kullanıcıya özel kasa bakiyesi
                $cashAccount = Account::getCashAccount($user->id);
                $cashBalance = $cashAccount->balance ?? 0;
                
                // Kasa hesabına özel transaction sorgusu
                $transactionQuery = Transaction::where('account_id', $cashAccount->id);
                if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                    $transactionQuery->where('created_by', $user->id);
                }
                
                $cashIncome = (clone $transactionQuery)->where('type', 'income')->sum('amount') ?? 0;
                $cashExpense = (clone $transactionQuery)->where('type', 'expense')->sum('amount') ?? 0;
                $todayIncome = (clone $transactionQuery)->where('type', 'income')->whereDate('date', $today)->sum('amount') ?? 0;
                $todayExpense = (clone $transactionQuery)->where('type', 'expense')->whereDate('date', $today)->sum('amount') ?? 0;
                
                // Bekleyen ödemeler (kalan bakiyesi olan rezervasyonlar)
                // remaining_amount alanı güncel olmayabilir, bu yüzden total_price - deposit_paid hesaplıyoruz
                $pendingQuery = Reservation::where('status', '!=', 'cancelled')
                    ->whereRaw('(total_price - COALESCE(deposit_paid, 0)) > 0');
                if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                    $pendingQuery->where('created_by', $user->id);
                }
                $pendingPayments = $pendingQuery->get()->sum(function($reservation) {
                    return max(0, ($reservation->total_price ?? 0) - ($reservation->deposit_paid ?? 0));
                });
            } catch (\Exception $e) {
                \Log::warning('Kasa bilgileri yüklenemedi: ' . $e->getMessage());
            }
            
            // Rezervasyoncu ise sadece kendi rezervasyonlarını göster
            $reservationQuery = Reservation::query();
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $reservationQuery->where('created_by', $user->id);
            }
            
            // Veritabanı kolonunu kontrol et
            $dateColumn = $this->getReservationDateColumn();
            
            $now = Carbon::now()->setTimezone('Europe/Istanbul');
            $stats = [
                'total_reservations' => (clone $reservationQuery)->count(),
                'today_count' => (clone $reservationQuery)->whereDate($dateColumn, $today)->count(),
                'month_count' => (clone $reservationQuery)->whereYear($dateColumn, $now->year)
                    ->whereMonth($dateColumn, $now->month)
                    ->count(),
                'open_count' => (clone $reservationQuery)->where('status', 'open')->count(),
                'confirmed_count' => (clone $reservationQuery)->where('status', 'confirmed')->count(),
                'cancelled_count' => (clone $reservationQuery)->where('status', 'cancelled')->count(),
                'completed_count' => (clone $reservationQuery)->where('status', 'completed')->count(),
                'month_revenue' => (clone $reservationQuery)->whereYear($dateColumn, $now->year)
                    ->whereMonth($dateColumn, $now->month)
                    ->where('status', '!=', 'cancelled')
                    ->sum('total_price'),
            ];

            // Grafik verileri - Farklı period'lar için
            $last7Days = [];
            $last30Days = [];
            $last12Months = [];
            $last12Weeks = [];
            
            // Son 7 gün (Günlük)
            for ($i = 6; $i >= 0; $i--) {
                $date = now()->setTimezone('Europe/Istanbul')->subDays($i)->format('Y-m-d');
                try {
                    $query = (clone $reservationQuery)->whereDate($dateColumn, $date);
                    $count = $query->count();
                } catch (\Exception $e) {
                    $count = 0;
                }
                $carbonDate = Carbon::now()->setTimezone('Europe/Istanbul')->subDays($i);
                $last7Days[] = [
                    'date' => $carbonDate->locale('tr')->format('d.m'),
                    'count' => $count,
                ];
            }
            
            // Son 30 gün (Günlük)
            for ($i = 29; $i >= 0; $i--) {
                $date = now()->setTimezone('Europe/Istanbul')->subDays($i)->format('Y-m-d');
                try {
                    $query = (clone $reservationQuery)->whereDate($dateColumn, $date);
                    $count = $query->count();
                } catch (\Exception $e) {
                    $count = 0;
                }
                $carbonDate = Carbon::now()->setTimezone('Europe/Istanbul')->subDays($i);
                $last30Days[] = [
                    'date' => $carbonDate->locale('tr')->format('d.m'),
                    'count' => $count,
                ];
            }
            
            // Son 12 hafta (Haftalık)
            for ($i = 11; $i >= 0; $i--) {
                $weekStart = now()->setTimezone('Europe/Istanbul')->subWeeks($i)->startOfWeek();
                $weekEnd = now()->setTimezone('Europe/Istanbul')->subWeeks($i)->endOfWeek();
                try {
                    $query = (clone $reservationQuery)
                        ->whereBetween($dateColumn, [$weekStart->format('Y-m-d'), $weekEnd->format('Y-m-d')]);
                    $count = $query->count();
                } catch (\Exception $e) {
                    $count = 0;
                }
                $last12Weeks[] = [
                    'date' => $weekStart->locale('tr')->format('d.m') . ' - ' . $weekEnd->locale('tr')->format('d.m'),
                    'count' => $count,
                ];
            }
            
            // Son 12 ay (Aylık)
            for ($i = 11; $i >= 0; $i--) {
                $month = now()->setTimezone('Europe/Istanbul')->subMonths($i);
                try {
                    $query = (clone $reservationQuery)
                        ->whereYear($dateColumn, $month->year)
                        ->whereMonth($dateColumn, $month->month);
                    $count = $query->count();
                } catch (\Exception $e) {
                    $count = 0;
                }
                $last12Months[] = [
                    'date' => $month->locale('tr')->translatedFormat('F Y'),
                    'count' => $count,
                ];
            }

            // Son rezervasyonlar
            try {
                $recent_reservations = (clone $reservationQuery)
                    ->with(['customer', 'salon'])
                    ->orderBy('created_at', 'desc')
                    ->take(10)
                    ->get();
            } catch (\Exception $e) {
                $recent_reservations = collect([]);
            }

            // Aylık gelir ve rezervasyon sayısı - SADECE KASA GELİRLERİ
            try {
                $now = Carbon::now()->setTimezone('Europe/Istanbul');
                
                // Kasa gelirleri (bu ay) - SADECE KASA
                $monthlyRevenue = 0;
                try {
                    $cashAccount = Account::getCashAccount($user->id);
                    $monthlyTransactionQuery = Transaction::where('account_id', $cashAccount->id)
                        ->where('type', 'income');
                    if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                        $monthlyTransactionQuery->where('created_by', $user->id);
                    }
                    $monthlyRevenue = $monthlyTransactionQuery
                        ->whereYear('date', $now->year)
                        ->whereMonth('date', $now->month)
                        ->sum('amount') ?? 0;
                } catch (\Exception $e) {
                    \Log::warning('Aylık kasa geliri hesaplanamadı: ' . $e->getMessage());
                }
                
                // stats['month_revenue'] ile tutarlılık için güncelle
                $stats['month_revenue'] = $monthlyRevenue;
                
                $monthlyReservations = (clone $reservationQuery)
                    ->whereYear($dateColumn, $now->year)
                    ->whereMonth($dateColumn, $now->month)
                    ->count();
            } catch (\Exception $e) {
                $monthlyRevenue = 0;
                $stats['month_revenue'] = 0;
                $monthlyReservations = 0;
            }

            // Haftalık rezervasyonlar (son 7 gün) - SADECE KASA GELİRLERİ
            try {
                $now = Carbon::now()->setTimezone('Europe/Istanbul');
                $weekStart = $now->copy()->startOfWeek()->toDateString();
                $weekEnd = $now->copy()->endOfWeek()->toDateString();
                
                $weeklyReservations = (clone $reservationQuery)
                    ->whereBetween($dateColumn, [$weekStart, $weekEnd])
                    ->count();

                // Haftalık kasa gelirleri - SADECE KASA
                $weeklyRevenue = 0;
                try {
                    $cashAccount = Account::getCashAccount($user->id);
                    $weeklyTransactionQuery = Transaction::where('account_id', $cashAccount->id)
                        ->where('type', 'income');
                    if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                        $weeklyTransactionQuery->where('created_by', $user->id);
                    }
                    $weeklyRevenue = $weeklyTransactionQuery
                        ->whereBetween('date', [$weekStart, $weekEnd])
                        ->sum('amount') ?? 0;
                } catch (\Exception $e) {
                    \Log::warning('Haftalık kasa geliri hesaplanamadı: ' . $e->getMessage());
                }
            } catch (\Exception $e) {
                $weeklyReservations = 0;
                $weeklyRevenue = 0;
            }

            // Ek istatistikler
            $totalCustomers = 0;
            $totalRevenue = 0;
            $averageReservationAmount = 0;
            $upcomingReservations = collect([]);
            $topCustomers = collect([]);
            $last30DaysRevenue = [];
            $lastMonthRevenue = 0;
            $lastMonthReservations = 0;
            
            try {
                // Toplam müşteri sayısı
                $customerQuery = Customer::query();
                if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                    $customerQuery->where('created_by', $user->id);
                }
                $totalCustomers = $customerQuery->count();
                
                // Toplam gelir - SADECE KASA GELİRLERİ
                $totalRevenue = 0;
                try {
                    $cashAccount = Account::getCashAccount($user->id);
                    $totalTransactionQuery = Transaction::where('account_id', $cashAccount->id)
                        ->where('type', 'income');
                    if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                        $totalTransactionQuery->where('created_by', $user->id);
                    }
                    $totalRevenue = $totalTransactionQuery->sum('amount') ?? 0;
                } catch (\Exception $e) {
                    \Log::warning('Toplam kasa geliri hesaplanamadı: ' . $e->getMessage());
                }
                
                // Ortalama rezervasyon tutarı
                $reservationCount = (clone $reservationQuery)->where('status', '!=', 'cancelled')->count();
                $averageReservationAmount = $reservationCount > 0 ? ($totalRevenue / $reservationCount) : 0;
                
                // Yaklaşan rezervasyonlar (önümüzdeki 7 gün)
                $now = Carbon::now()->setTimezone('Europe/Istanbul');
                $upcomingReservations = (clone $reservationQuery)
                    ->with(['customer', 'salon'])
                    ->whereDate($dateColumn, '>=', $today)
                    ->whereDate($dateColumn, '<=', $now->addDays(7)->format('Y-m-d'))
                    ->where('status', '!=', 'cancelled')
                    ->orderBy($dateColumn, 'asc')
                    ->take(5)
                    ->get();
                
                // En çok rezervasyon yapan müşteriler
                $topCustomers = (clone $reservationQuery)
                    ->select('customer_id', DB::raw('count(*) as reservation_count'), DB::raw('sum(total_price) as total_spent'))
                    ->where('status', '!=', 'cancelled')
                    ->groupBy('customer_id')
                    ->orderBy('reservation_count', 'desc')
                    ->take(5)
                    ->get()
                    ->map(function($item) {
                        $customer = Customer::find($item->customer_id);
                        return [
                            'customer' => $customer,
                            'reservation_count' => $item->reservation_count,
                            'total_spent' => $item->total_spent ?? 0
                        ];
                    })
                    ->filter(function($item) {
                        return $item['customer'] !== null;
                    });
                
                // Son 30 gün gelir grafiği için
                for ($i = 29; $i >= 0; $i--) {
                    $date = now()->setTimezone('Europe/Istanbul')->subDays($i)->format('Y-m-d');
                    $dayRevenue = (clone $reservationQuery)
                        ->whereDate($dateColumn, $date)
                        ->where('status', '!=', 'cancelled')
                        ->sum('total_price');
                    $carbonDate = Carbon::now()->setTimezone('Europe/Istanbul')->subDays($i);
                    $last30DaysRevenue[] = [
                        'date' => $carbonDate->locale('tr')->format('d.m'),
                        'revenue' => $dayRevenue,
                    ];
                }
                
                // Geçen ay gelir ve rezervasyon
                $lastMonth = Carbon::now()->setTimezone('Europe/Istanbul')->subMonth();
                $lastMonthRevenue = (clone $reservationQuery)
                    ->whereYear($dateColumn, $lastMonth->year)
                    ->whereMonth($dateColumn, $lastMonth->month)
                    ->where('status', '!=', 'cancelled')
                    ->sum('total_price');
                
                $lastMonthReservations = (clone $reservationQuery)
                    ->whereYear($dateColumn, $lastMonth->year)
                    ->whereMonth($dateColumn, $lastMonth->month)
                    ->count();
            } catch (\Exception $e) {
                \Log::warning('Ek istatistikler yüklenemedi: ' . $e->getMessage());
            }

            return view('admin.dashboard', compact(
                'stats', 
                'last7Days',
                'last30Days',
                'last12Weeks',
                'last12Months',
                'recent_reservations', 
                'monthlyRevenue', 
                'monthlyReservations', 
                'weeklyReservations', 
                'weeklyRevenue',
                'cashBalance',
                'cashIncome',
                'cashExpense',
                'todayIncome',
                'todayExpense',
                'pendingPayments',
                'showSetupWizard',
                'totalCustomers',
                'totalRevenue',
                'averageReservationAmount',
                'upcomingReservations',
                'topCustomers',
                'last30DaysRevenue',
                'lastMonthRevenue',
                'lastMonthReservations'
            ));
            
        } catch (\Exception $e) {
            \Log::error('Dashboard Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            
            // Hata durumunda boş verilerle göster
            $stats = [
                'total_reservations' => 0,
                'today_count' => 0,
                'month_count' => 0,
                'open_count' => 0,
                'confirmed_count' => 0,
                'cancelled_count' => 0,
                'month_revenue' => 0,
            ];
            $last7Days = [];
            $last30Days = [];
            $last12Weeks = [];
            $last12Months = [];
            $recent_reservations = collect([]);
            $monthlyRevenue = 0;
            $monthlyReservations = 0;
            $weeklyReservations = 0;
            $weeklyRevenue = 0;
            $cashBalance = 0;
            $cashIncome = 0;
            $cashExpense = 0;
            $todayIncome = 0;
            $todayExpense = 0;
            $pendingPayments = 0;
            $showSetupWizard = false;
            $totalCustomers = 0;
            $totalRevenue = 0;
            $averageReservationAmount = 0;
            $upcomingReservations = collect([]);
            $topCustomers = collect([]);
            $last30DaysRevenue = [];
            $lastMonthRevenue = 0;
            $lastMonthReservations = 0;
            
            return view('admin.dashboard', compact(
                'stats', 
                'last7Days',
                'last30Days',
                'last12Weeks',
                'last12Months',
                'recent_reservations', 
                'monthlyRevenue', 
                'monthlyReservations', 
                'weeklyReservations', 
                'weeklyRevenue',
                'cashBalance',
                'cashIncome',
                'cashExpense',
                'todayIncome',
                'todayExpense',
                'pendingPayments',
                'showSetupWizard',
                'totalCustomers',
                'totalRevenue',
                'averageReservationAmount',
                'upcomingReservations',
                'topCustomers',
                'last30DaysRevenue',
                'lastMonthRevenue',
                'lastMonthReservations'
            ))->with('error', 'Bazı veriler yüklenirken hata oluştu. Lütfen sayfayı yenileyin.');
        }
    }
}

