<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Invoice;
use App\Models\Reservation;
use App\Models\Setting;
use App\Services\SmsService;
use App\Services\WhatsAppService;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class InvoiceController extends Controller
{
    /**
     * Fatura PDF'i oluştur ve göster (Rezervasyon ID ile)
     */
    public function show($id)
    {
        $reservation = Reservation::with(['customer', 'salon', 'extras'])->findOrFail($id);
        
        // Ayarları al
        $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
        $logo = $settings['invoice_logo'] ?? null;
        $address = $settings['invoice_address'] ?? '';
        $taxOffice = $settings['invoice_tax_office'] ?? '';
        $taxNumber = $settings['invoice_tax_number'] ?? '';
        
        // Salon adresi (öncelik salon adresi, yoksa genel adres)
        $salonAddress = $reservation->salon->address ?? $address;
        
        // Fatura verilerini hazırla
        $invoiceData = $this->prepareInvoiceData($reservation, $logo, $salonAddress, $taxOffice, $taxNumber);
        
        // PDF oluştur
        $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);
        
        return $pdf->stream('fatura-' . $reservation->id . '.pdf');
    }
    
    /**
     * Fatura PDF'i göster (Invoice ID ile)
     */
    public function showPdf($id)
    {
        $user = Auth::user();
        $invoice = Invoice::with(['reservation.customer', 'reservation.salon', 'reservation.extras', 'customer'])->findOrFail($id);
        
        // Erişim kontrolü: Sadece kendi faturalarını veya rezervasyoncular kendi rezervasyonlarının faturalarını görebilir
        if ($user->role !== 'super_admin') {
            if ($user->role === 'rezervasyoncu') {
                // Rezervasyoncu ise, rezervasyonun kendisine ait olup olmadığını kontrol et
                if ($invoice->reservation && $invoice->reservation->created_by !== $user->id) {
                    abort(403, 'Bu faturaya erişim yetkiniz yok.');
                }
            } else {
                // Admin ise, faturayı kendisi oluşturmuş olmalı
                if ($invoice->created_by !== $user->id) {
                    abort(403, 'Bu faturaya erişim yetkiniz yok.');
                }
            }
        }
        
        // Ayarları al
        $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
        $generalSettings = Setting::where('group', 'general')->pluck('value', 'key');
        $logo = $settings['invoice_logo'] ?? null;
        $address = $settings['invoice_address'] ?? '';
        $taxOffice = $settings['invoice_tax_office'] ?? '';
        $taxNumber = $settings['invoice_tax_number'] ?? '';
        $companyName = $generalSettings['company_name'] ?? 'Adugun';
        
        // Transaction faturaları için reservation null olabilir
        if (!$invoice->reservation) {
            // Transaction faturaları için PDF oluştur
            $invoiceData = $this->prepareTransactionInvoiceData($invoice, $logo, $address, $taxOffice, $taxNumber, $companyName);
        } else {
            // Rezervasyon faturaları için mevcut mantık
            $reservation = $invoice->reservation;
            $salonAddress = $reservation->salon->address ?? $address;
            $invoiceData = $this->prepareInvoiceData($reservation, $logo, $salonAddress, $taxOffice, $taxNumber, $invoice);
        }
        
        // PDF oluştur
        $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);
        
        return $pdf->stream('fatura-' . $invoice->invoice_number . '.pdf');
    }
    
    /**
     * Fatura PDF'i indir (Invoice ID ile)
     */
    public function downloadPdf($id)
    {
        $user = Auth::user();
        $invoice = Invoice::with(['reservation.customer', 'reservation.salon', 'reservation.extras', 'customer'])->findOrFail($id);
        
        // Erişim kontrolü: Sadece kendi faturalarını veya rezervasyoncular kendi rezervasyonlarının faturalarını görebilir
        if ($user->role !== 'super_admin') {
            if ($user->role === 'rezervasyoncu') {
                // Rezervasyoncu ise, rezervasyonun kendisine ait olup olmadığını kontrol et
                if ($invoice->reservation && $invoice->reservation->created_by !== $user->id) {
                    abort(403, 'Bu faturaya erişim yetkiniz yok.');
                }
            } else {
                // Admin ise, faturayı kendisi oluşturmuş olmalı
                if ($invoice->created_by !== $user->id) {
                    abort(403, 'Bu faturaya erişim yetkiniz yok.');
                }
            }
        }
        
        // Ayarları al
        $settings = Setting::where('group', 'invoice')->pluck('value', 'key');
        $generalSettings = Setting::where('group', 'general')->pluck('value', 'key');
        $logo = $settings['invoice_logo'] ?? null;
        $address = $settings['invoice_address'] ?? '';
        $taxOffice = $settings['invoice_tax_office'] ?? '';
        $taxNumber = $settings['invoice_tax_number'] ?? '';
        $companyName = $generalSettings['company_name'] ?? 'Adugun';
        
        // Transaction faturaları için reservation null olabilir
        if (!$invoice->reservation) {
            // Transaction faturaları için PDF oluştur
            $invoiceData = $this->prepareTransactionInvoiceData($invoice, $logo, $address, $taxOffice, $taxNumber, $companyName);
        } else {
            // Rezervasyon faturaları için mevcut mantık
            $reservation = $invoice->reservation;
            $salonAddress = $reservation->salon->address ?? $address;
            $invoiceData = $this->prepareInvoiceData($reservation, $logo, $salonAddress, $taxOffice, $taxNumber, $invoice);
        }
        
        // PDF oluştur ve indir
        $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);
        
        return $pdf->download('fatura-' . $invoice->invoice_number . '.pdf');
    }
    
    /**
     * Fatura SMS gönder
     */
    public function sendSms(Request $request, $id)
    {
        $invoice = Invoice::with(['reservation.customer'])->findOrFail($id);
        
        if (!$invoice->reservation->customer->phone) {
            return response()->json([
                'success' => false,
                'message' => 'Müşterinin telefon numarası bulunamadı.'
            ], 400);
        }
        
        // SMS bakiyesi kontrolü
        $user = Auth::user();
        if ($user && !$user->isSuperAdmin()) {
            if (!$user->hasSmsBalance(1)) {
                return response()->json([
                    'success' => false,
                    'message' => 'SMS bakiyeniz kalmamıştır. Lütfen SMS paketi satın alın.',
                    'redirect' => route('admin.sms-packages.index')
                ], 400);
            }
        }
        
        $smsService = app(\App\Services\SmsService::class);
        $invoiceUrl = $invoice->public_url;
        
        // SMS gönder (bakiye direkt düşecek - skip_balance_deduction false olarak açıkça belirtildi)
        $message = "Fatura linkiniz: {$invoiceUrl}";
        $result = $smsService->send(
            $invoice->reservation->customer->phone,
            $message,
            null,
            [
                'purpose' => 'invoice', 
                'user_id' => $user->id,
                'skip_balance_deduction' => false // SMS kredisi direkt düşürülecek
            ]
        );
        
            if ($result->success) {
                // Bakiyeyi yeniden yükle
                $user->refresh();
            
            return response()->json([
                'success' => true,
                'message' => 'SMS başarıyla gönderildi.',
                'remaining_balance' => $user && !$user->isSuperAdmin() ? $user->sms_balance : null
            ]);
        } else {
            return response()->json([
                'success' => false,
                'message' => 'SMS gönderilirken bir hata oluştu: ' . ($result->errorMessage ?? 'Bilinmeyen hata')
            ], 500);
        }
    }
    
    /**
     * WhatsApp ile fatura linki gönder
     */
    public function sendWhatsApp(Request $request, $id)
    {
        $invoice = Invoice::with(['reservation.customer', 'reservation.salon'])->findOrFail($id);
        
        if (!$invoice->reservation->customer->phone) {
            return response()->json([
                'success' => false,
                'message' => 'Müşterinin telefon numarası bulunamadı.'
            ], 400);
        }
        
        try {
            // Varsayılan sözleşme linkini al
            $defaultContract = Contract::getDefaultForCompany(auth()->user()->company_id);
            $contractUrl = $defaultContract ? $defaultContract->public_url : null;
            
            // Rezervasyon tarih/saat bilgisini hazırla
            $reservation = $invoice->reservation;
            $reservationDate = $reservation->date ? $reservation->date->format('Y-m-d') : null;
            $reservationTime = '';
            if ($reservation->start_time && $reservation->end_time) {
                $reservationTime = $reservation->start_time . ' - ' . $reservation->end_time;
            } elseif ($reservation->start_time) {
                $reservationTime = $reservation->start_time;
            }
            
            // MessageService ile gönder (sadece WhatsApp)
            $messageService = app(\App\Services\MessageService::class);
            $result = $messageService->sendInvoiceAndContract(
                phone: $invoice->reservation->customer->phone,
                invoiceUrl: $invoice->public_url,
                contractUrl: $contractUrl,
                salonId: $invoice->reservation->salon_id,
                reservationDate: $reservationDate,
                reservationTime: $reservationTime,
                customerName: $invoice->reservation->customer->name
            );
            
            if ($result) {
                return response()->json([
                    'success' => true,
                    'message' => 'Mesaj başarıyla gönderildi (WhatsApp veya SMS).'
                ]);
            }
        } catch (\Exception $e) {
            \Log::error('Mesaj gönderim hatası: ' . $e->getMessage());
        }
        
        return response()->json([
            'success' => false,
            'message' => 'Mesaj gönderilemedi. API ayarlarınızı kontrol edin.'
        ], 500);
    }
    
    
    /**
     * Faturayı sil
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $query = Invoice::query();
        
        // Süper admin değilse sadece kendi faturalarını silebilir
        if ($user->role !== 'super_admin') {
            $query->where('created_by', $user->id);
        }
        
        $invoice = $query->findOrFail($id);
        
        // PDF dosyasını da sil
        if ($invoice->pdf_path && Storage::disk('public')->exists($invoice->pdf_path)) {
            Storage::disk('public')->delete($invoice->pdf_path);
        }
        
        $invoice->delete();
        
        if (request()->ajax() || request()->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Fatura başarıyla silindi.'
            ]);
        }
        
        return redirect()->route('admin.settings.invoices.index')
            ->with('success', 'Fatura başarıyla silindi.');
    }
    
    /**
     * Public URL oluştur (localhost'u adugun.com'a çevir)
     */
    private function getPublicUrl($token)
    {
        $url = url('/fatura/' . $token);
        // Eğer localhost içeriyorsa adugun.com'a çevir
        if (strpos($url, 'localhost') !== false || strpos($url, '127.0.0.1') !== false) {
            $url = str_replace(['http://localhost', 'https://localhost', 'http://127.0.0.1', 'https://127.0.0.1'], 'https://adugun.com', $url);
        }
        return $url;
    }

    /**
     * Fatura verilerini hazırla
     */
    private function prepareInvoiceData($reservation, $logo, $address, $taxOffice, $taxNumber, $invoice = null)
    {
        // Yemekleri, organizasyonu ve ekstraları topla
        $items = [];
        $itemNumber = 0;
        
        // items_data veya notes'tan JSON verilerini çıkar
        $itemsData = json_decode($reservation->items_data ?? '{}', true);
        if (!is_array($itemsData) || empty($itemsData)) {
            $itemsData = json_decode($reservation->notes ?? '{}', true);
            if (!is_array($itemsData)) {
                $itemsData = [];
            }
        }
        
        // Toplam fiyattan parçaları çıkararak salon ücretini bul
        $mealTotal = 0;
        $organizationPrice = $itemsData['organization_price'] ?? 0;
        $extraTotal = 0;
        
        // Yemek toplamı
        if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
            foreach ($itemsData['meals'] as $meal) {
                if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                    $total = $meal['quantity'] * $meal['price'];
                    $mealTotal += $total;
                }
            }
        }
        
        // Ekstra toplamı
        if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
            foreach ($itemsData['extras'] as $extra) {
                if (isset($extra['name']) && isset($extra['price']) && $extra['price'] > 0) {
                    $extraTotal += $extra['price'];
                }
            }
        }
        
        // Salon ücreti = Toplam - (Yemekler + Organizasyon + Ekstralar)
        $salonPrice = $reservation->total_price - ($mealTotal + $organizationPrice + $extraTotal);
        
        // 1. Salon ücretini başa ekle
        if ($salonPrice > 0) {
            $items[] = [
                'name' => 'Salon Ucreti',
                'quantity' => '-',
                'price' => $salonPrice,
                'total' => $salonPrice
            ];
        }
        
        // 2. Yemekler
        if (isset($itemsData['meals']) && is_array($itemsData['meals'])) {
            foreach ($itemsData['meals'] as $meal) {
                if (isset($meal['name']) && isset($meal['quantity']) && isset($meal['price'])) {
                    $total = $meal['quantity'] * $meal['price'];
                    $items[] = [
                        'name' => $meal['name'],
                        'quantity' => $meal['quantity'] . ' Kisi',
                        'price' => $meal['price'],
                        'total' => $total
                    ];
                }
            }
        }
        
        // 3. Organizasyon
        if ($organizationPrice > 0) {
            $items[] = [
                'name' => 'Organizasyon',
                'quantity' => '-',
                'price' => $organizationPrice,
                'total' => $organizationPrice
            ];
        }
        
        // 4. Ekstralar
        if (isset($itemsData['extras']) && is_array($itemsData['extras'])) {
            foreach ($itemsData['extras'] as $extra) {
                if (isset($extra['name']) && isset($extra['price']) && $extra['price'] > 0) {
                    $items[] = [
                        'name' => $extra['name'],
                        'quantity' => '-',
                        'price' => $extra['price'],
                        'total' => $extra['price']
                    ];
                }
            }
        }
        
        // Toplam fiyat ve kapora
        $grandTotal = $reservation->total_price;
        $depositPaid = $reservation->deposit_paid ?? 0;
        
        // Salon adı ve adresi
        $salonName = $reservation->salon->name ?? 'Dugun Salonu';
        $salonAddress = $reservation->salon->address ?? $address;
        
        return [
            'reservation' => $reservation,
            'invoice' => $invoice,
            'items' => $items,
            'subtotal' => $grandTotal,
            'grandTotal' => $grandTotal,
            'depositPaid' => $depositPaid,
            'logo' => $logo,
            'address' => $address,
            'salonName' => $salonName,
            'salonAddress' => $salonAddress,
            'taxOffice' => $taxOffice,
            'taxNumber' => $taxNumber,
        ];
    }
    
    /**
     * Transaction faturaları için fatura verilerini hazırla
     */
    private function prepareTransactionInvoiceData($invoice, $logo, $address, $taxOffice, $taxNumber, $companyName)
    {
        // Invoice'dan items_json'ı al
        $items = $invoice->items_json ?? [];
        
        // Metadata'yı temizle (sadece görüntüleme için gerekli alanlar)
        $cleanItems = [];
        foreach ($items as $item) {
            if (is_array($item)) {
                $cleanItems[] = [
                    'name' => $item['name'] ?? 'İşlem',
                    'quantity' => $item['quantity'] ?? '-',
                    'price' => $item['price'] ?? 0,
                    'total' => $item['total'] ?? ($item['price'] ?? 0),
                ];
            }
        }
        
        // Eğer items boşsa, invoice'dan bilgileri al
        if (empty($cleanItems)) {
            $cleanItems[] = [
                'name' => 'İşlem',
                'quantity' => '-',
                'price' => $invoice->total_amount,
                'total' => $invoice->total_amount,
            ];
        }
        
        // Müşteri bilgisi
        $customerName = $invoice->customer ? $invoice->customer->name : 'Müşteri';
        
        // Fatura tarihi
        $invoiceDate = $invoice->invoice_date ?? $invoice->created_at;
        
        return [
            'reservation' => null,
            'transaction' => true,
            'invoice' => $invoice,
            'items' => $cleanItems,
            'subtotal' => $invoice->total_amount,
            'grandTotal' => $invoice->total_amount,
            'depositPaid' => $invoice->deposit_paid ?? 0,
            'logo' => $logo,
            'address' => $address,
            'salonName' => $companyName,
            'salonAddress' => $address,
            'taxOffice' => $taxOffice,
            'taxNumber' => $taxNumber,
            'customerName' => $customerName,
            'invoiceDate' => $invoiceDate,
        ];
    }

    /**
     * Fatura kaydet
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'reservation_id' => 'required|exists:reservations,id',
        ]);

        $reservation = Reservation::with(['customer', 'salon'])->findOrFail($validated['reservation_id']);
        
        // Fatura numarası oluştur
        $invoiceNumber = 'FAT-' . date('Y') . '-' . str_pad(Invoice::count() + 1, 6, '0', STR_PAD_LEFT);
        
        // Public token oluştur
        $publicToken = Str::random(32);
        
        // Fatura kalemlerini hazırla
        $invoiceData = $this->prepareInvoiceData(
            $reservation,
            Setting::where('group', 'invoice')->where('key', 'invoice_logo')->value('value'),
            $reservation->salon->address ?? Setting::where('group', 'invoice')->where('key', 'invoice_address')->value('value') ?? '',
            Setting::where('group', 'invoice')->where('key', 'invoice_tax_office')->value('value') ?? '',
            Setting::where('group', 'invoice')->where('key', 'invoice_tax_number')->value('value') ?? ''
        );
        
        // PDF oluştur
        $pdf = Pdf::loadView('admin.invoices.pdf', $invoiceData);
        
        // PDF'i kaydet
        $pdfPath = 'invoices/' . $invoiceNumber . '.pdf';
        Storage::disk('public')->put($pdfPath, $pdf->output());
        
        // Faturayı veritabanına kaydet
        $invoice = Invoice::create([
            'reservation_id' => $reservation->id,
            'customer_id' => $reservation->customer_id,
            'invoice_number' => $invoiceNumber,
            'total_amount' => $reservation->total_price,
            'deposit_paid' => $reservation->deposit_paid ?? 0,
            'remaining_amount' => $reservation->remaining_amount ?? ($reservation->total_price - ($reservation->deposit_paid ?? 0)),
            'invoice_date' => now(),
            'pdf_path' => $pdfPath,
            'public_token' => $publicToken,
            'public_url' => $this->getPublicUrl($publicToken),
            'status' => 'sent', // Direkt gönderildi olarak oluştur
            'items_json' => $invoiceData['items'],
            'created_by' => Auth::id(),
        ]);
        
        // SMS gönder (eğer istendiyse)
        if ($request->has('send_sms') && $request->send_sms == '1' && $reservation->customer->phone) {
            $user = Auth::user();
            if ($user && !$user->isSuperAdmin()) {
                if ($user->hasSmsBalance(1)) {
                    $smsService = app(\App\Services\SmsService::class);
                    $invoiceUrl = $invoice->public_url; // Görüntüleme sayfası linki
                    $message = "Fatura linkiniz: {$invoiceUrl}";
                    $smsService->send(
                        $reservation->customer->phone,
                        $message,
                        null,
                        ['purpose' => 'invoice', 'user_id' => $user->id]
                    );
                }
            } else {
                $smsService = app(\App\Services\SmsService::class);
                $invoiceUrl = $invoice->public_url; // Görüntüleme sayfası linki
                $message = "Fatura linkiniz: {$invoiceUrl}";
                $smsService->send(
                    $reservation->customer->phone,
                    $message,
                    null,
                    ['purpose' => 'invoice', 'user_id' => $user->id]
                );
            }
        }
        
        return response()->json([
            'success' => true,
            'message' => 'Fatura başarıyla oluşturuldu.',
            'invoice_id' => $invoice->id,
            'invoice_number' => $invoice->invoice_number,
            'public_url' => $invoice->public_url,
        ]);
    }

    /**
     * Tüm faturaları listele
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $query = Invoice::with(['reservation', 'customer', 'creator']);
        
        // Sıralama - Önce sıralamayı yap, sonra filtreleme
        $sortBy = $request->get('sort', 'created_desc'); // Default: Yeniden Eskiye
        $needsJoin = ($sortBy === 'name_asc');
        
        if ($needsJoin) {
            // A-Z (müşteri adına göre) - Join yapılacak
            $query->leftJoin('customers', 'invoices.customer_id', '=', 'customers.id')
                  ->select('invoices.*');
        }
        
        // Tarih filtresi (invoice_date kolonuna göre)
        if ($request->filled('start_date')) {
            $startDate = $request->get('start_date');
            if ($needsJoin) {
                $query->whereDate('invoices.invoice_date', '>=', $startDate);
            } else {
                $query->whereDate('invoice_date', '>=', $startDate);
            }
        }
        
        if ($request->filled('end_date')) {
            $endDate = $request->get('end_date');
            if ($needsJoin) {
                $query->whereDate('invoices.invoice_date', '<=', $endDate);
            } else {
                $query->whereDate('invoice_date', '<=', $endDate);
            }
        }
        
        // Süper admin değilse sadece kendi faturalarını göster
        if ($user->role !== 'super_admin') {
            if ($user->role === 'rezervasyoncu') {
                // Rezervasyoncu ise, kendi rezervasyonlarına ait faturaları VEYA kendi oluşturduğu faturaları göster
                // (rezervasyon olmasa bile - transaction faturaları için)
                $query->where(function($q) use ($user, $needsJoin) {
                    // Kendi oluşturduğu tüm faturalar (rezervasyon olsun veya olmasın)
                    if ($needsJoin) {
                        $q->where('invoices.created_by', $user->id);
                    } else {
                        $q->where('created_by', $user->id);
                    }
                });
            } else {
                // Admin ise, kendi oluşturduğu faturaları göster (rezervasyon olsun veya olmasın)
                // Join yapıldıysa tablo adını belirt
                if ($needsJoin) {
                    $query->where('invoices.created_by', $user->id);
                } else {
                    $query->where('created_by', $user->id);
                }
            }
        }
        
        // Sıralama uygula
        switch ($sortBy) {
            case 'name_asc':
                // A-Z (müşteri adına göre)
                $query->orderBy('customers.name', 'asc');
                break;
            case 'created_desc':
                // Yeniden Eskiye (oluşturulma tarihine göre desc)
                if ($needsJoin) {
                    $query->orderBy('invoices.created_at', 'desc');
                } else {
                    $query->orderBy('created_at', 'desc');
                }
                break;
            default:
                if ($needsJoin) {
                    $query->orderBy('invoices.created_at', 'desc');
                } else {
                    $query->orderBy('created_at', 'desc');
                }
        }
        
        $invoices = $query->paginate(10);
        
        // Toplam tutarı hesapla (tüm faturalar için, sayfalama olmadan)
        $totalAmountQuery = Invoice::query();
        if ($user->role !== 'super_admin') {
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $totalAmountQuery->where('created_by', $user->id);
            }
        }
        $totalAmount = $totalAmountQuery->sum('total_amount');
        
        return view('admin.settings.invoices.index', compact('invoices', 'totalAmount'));
    }
}
