<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Reservation;
use Carbon\Carbon;

class NotificationController extends Controller
{
    public function index()
    {
        $notifications = [];
        
        // Yeni rezervasyonlar
        $newReservations = Reservation::where('created_at', '>=', Carbon::now()->subDays(7))
            ->with(['customer', 'salon'])
            ->orderBy('created_at', 'desc')
            ->get();
            
        foreach ($newReservations as $reservation) {
            $notifications[] = [
                'type' => 'new_reservation',
                'icon' => 'calendar-check',
                'color' => 'primary',
                'title' => 'Yeni rezervasyon oluşturuldu',
                'message' => $reservation->customer->name . ' için ' . $reservation->salon->name . ' rezervasyonu',
                'date' => $reservation->created_at,
                'url' => route('admin.rezervasyonlar.show', $reservation->id)
            ];
        }
        
        // Yaklaşan rezervasyonlar (3 gün içinde)
        $upcomingReservations = Reservation::whereBetween('date', [
                Carbon::now()->toDateString(),
                Carbon::now()->addDays(3)->toDateString()
            ])
            ->where('status', '!=', 'cancelled')
            ->with(['customer', 'salon'])
            ->orderBy('date', 'asc')
            ->get();
            
        foreach ($upcomingReservations as $reservation) {
            $notifications[] = [
                'type' => 'upcoming',
                'icon' => 'clock',
                'color' => 'warning',
                'title' => 'Yaklaşan rezervasyon',
                'message' => $reservation->customer->name . ' - ' . $reservation->salon->name . ' (' . Carbon::parse($reservation->date)->format('d.m.Y') . ')',
                'date' => $reservation->date,
                'url' => route('admin.rezervasyonlar.show', $reservation->id)
            ];
        }
        
        // Sıralama
        usort($notifications, function($a, $b) {
            return $b['date'] <=> $a['date'];
        });
        
        return view('admin.notifications.index', compact('notifications'));
    }
    
    /**
     * Bugünkü rezervasyonları JSON olarak döndür
     */
    public function todayReservations(Request $request)
    {
        try {
            $user = auth()->user();
            $today = Carbon::today()->toDateString();
            
            $query = Reservation::with(['customer', 'salon'])
                ->where('date', $today)
                ->where('status', '!=', 'cancelled')
                ->orderBy('start_time', 'asc');
            
            // Rezervasyoncu ve admin ise sadece kendi rezervasyonlarını göster
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
            
            $reservations = $query->get();
            
            return response()->json([
                'reservations' => $reservations,
                'count' => $reservations->count()
            ]);
        } catch (\Exception $e) {
            \Log::error('Today Reservations Error: ' . $e->getMessage());
            return response()->json([
                'reservations' => [],
                'count' => 0,
                'error' => 'Rezervasyonlar yüklenirken bir hata oluştu.'
            ], 500);
        }
    }
    
    /**
     * Yaklaşan rezervasyonları JSON olarak döndür (bildirimler için)
     */
    public function upcomingReservations(Request $request)
    {
        try {
            $user = auth()->user();
            $now = Carbon::now();
            $oneHourLater = $now->copy()->addHour();
            $thirtyMinutesLater = $now->copy()->addMinutes(30);
            
            $query = Reservation::with(['customer', 'salon'])
                ->where('status', '!=', 'cancelled')
                ->where('status', '!=', 'completed')
                ->where('date', '>=', $now->toDateString());
            
            // Rezervasyoncu ve admin ise sadece kendi rezervasyonlarını göster
            if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
                $query->where('created_by', $user->id);
            }
            
            $reservations = $query->get()->filter(function($reservation) use ($now, $oneHourLater) {
                try {
                    if (!$reservation->date || !$reservation->start_time) {
                        return false;
                    }
                    
                    $reservationDateTime = Carbon::parse($reservation->date . ' ' . $reservation->start_time);
                    
                    // Rezervasyon şu andan sonra ve 1 saat içinde olmalı
                    if ($reservationDateTime < $now) {
                        return false;
                    }
                    
                    if ($reservationDateTime > $oneHourLater) {
                        return false;
                    }
                    
                    return true;
                } catch (\Exception $e) {
                    \Log::warning('Rezervasyon tarih/saat parse hatası: ' . $e->getMessage(), [
                        'reservation_id' => $reservation->id ?? null,
                        'date' => $reservation->date ?? null,
                        'start_time' => $reservation->start_time ?? null
                    ]);
                    return false;
                }
            });
            
            return response()->json([
                'reservations' => $reservations->values()->map(function($reservation) {
                    return [
                        'id' => $reservation->id,
                        'date' => $reservation->date,
                        'start_time' => $reservation->start_time,
                        'end_time' => $reservation->end_time,
                        'status' => $reservation->status,
                        'customer' => $reservation->customer ? [
                            'id' => $reservation->customer->id,
                            'name' => $reservation->customer->name,
                            'phone' => $reservation->customer->phone
                        ] : null,
                        'salon' => $reservation->salon ? [
                            'id' => $reservation->salon->id,
                            'name' => $reservation->salon->name
                        ] : null
                    ];
                })
            ]);
        } catch (\Exception $e) {
            \Log::error('Upcoming Reservations Error: ' . $e->getMessage());
            return response()->json([
                'reservations' => [],
                'error' => 'Yaklaşan rezervasyonlar yüklenirken bir hata oluştu.'
            ], 500);
        }
    }
}
