<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use Illuminate\Http\Request;

class ReservationCalendarController extends Controller
{
    public function index(Request $request)
    {
        $user = auth()->user();
        $query = \App\Models\Salon::where('is_active', true);
        
        // Rezervasyoncu veya admin ise sadece kendi salonlarını göster
        if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $salons = $query->get();
        return view('admin.reservations.calendar', compact('salons'));
    }

    public function feed(Request $request)
    {
        $start = $request->get('start');
        $end = $request->get('end');
        $salonId = $request->get('salon_id');

        $user = auth()->user();
        $query = Reservation::with(['customer', 'salon'])
            ->whereBetween('date', [$start, $end])
            ->whereNotIn('status', ['cancelled', 'completed']); // Sadece aktif rezervasyonlar (iptal ve tamamlanan hariç)

        // Rezervasyoncu veya admin ise sadece kendi rezervasyonlarını göster
        if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }

        if ($salonId) {
            $query->where('salon_id', $salonId);
        }

        $reservations = $query->get()->map(function($reservation) {
            // Event rengini status'a göre ayarla
            $color = '#3b82f6'; // default blue
            switch($reservation->status) {
                case 'confirmed':
                    $color = '#3b82f6'; // blue
                    break;
                case 'open':
                    $color = '#f59e0b'; // orange
                    break;
                case 'completed':
                    $color = '#22c55e'; // green
                    break;
                case 'cancelled':
                    $color = '#ef4444'; // red
                    break;
            }

            return [
                'id' => $reservation->id,
                'title' => ($reservation->customer->name ?? 'Müşteri') . ' - ' . ($reservation->salon->name ?? 'Salon'),
                'start' => $reservation->date->format('Y-m-d') . 'T' . $reservation->start_time,
                'end' => $reservation->date->format('Y-m-d') . 'T' . $reservation->end_time,
                'color' => $color,
                'extendedProps' => [
                    'status' => $reservation->status,
                    'customer' => $reservation->customer->name ?? '',
                    'salon' => $reservation->salon->name ?? '',
                ],
                'url' => route('admin.rezervasyonlar.show', $reservation->id),
            ];
        });

        return response()->json($reservations);
    }

    public function year()
    {
        $user = auth()->user();
        $query = Reservation::select('date', 'status')
            ->whereYear('date', now()->year);
        
        // Rezervasyoncu ise sadece kendi rezervasyonlarını göster
        if ($user->role === 'rezervasyoncu') {
            $query->where('created_by', $user->id);
        }
        
        $reservations = $query->get()
            ->groupBy(function($item) {
                return $item->date->format('Y-m-d');
            });

        return view('admin.reservations.year-calendar', compact('reservations'));
    }

    public function yearV2()
    {
        $user = auth()->user();
        $query = Reservation::with(['customer', 'salon'])
            ->whereYear('date', now()->year);
        
        // Rezervasyoncu veya admin ise sadece kendi rezervasyonlarını göster
        if ($user->role === 'rezervasyoncu' || $user->role === 'admin') {
            $query->where('created_by', $user->id);
        }
        
        $reservations = $query->get()
            ->groupBy(function($item) {
                return $item->date->format('Y-m-d');
            });

        return view('admin.reservations.year-calendar-v2', compact('reservations'));
    }
}

